package jp.kitec.lib.io;

import java.io.PrintStream;
import java.util.List;
import java.util.StringTokenizer;

import jp.kitec.lib.util.tree.ObjectFolder;
import jp.kitec.lib.util.tree.ObjectNode;
import jp.kitec.lib.util.xml.XMLManager;
import jp.kitec.lib.util.xml.XMLWriter;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;


/**
 * @author kawae
 */
public class FileSystem {

	private static final Log log = LogFactory.getLog(FileSystem.class);

	/** ディレクトリセパレータ */
	protected static final String SEPARATOR_CANDIDATE = "\\/";
	protected static final char SEPARATOR = '/';

	/** トップフォルダ */
	protected ObjectFolder mRootFolder;

	/**
	 * コンストラクタ
	 */
	public FileSystem() {
		mRootFolder = new ObjectFolder("ROOT");
	}

	/**
	 * rootフォルダの取得
	 * @return
	 */
	public ObjectFolder getRootFolder() {
		return mRootFolder;
	}

	/**
	 * ターゲットノードの取得
	 * @param s		パス
	 * @return
	 */
	public ObjectNode getTargetNode(String s) {
		StringTokenizer st = new StringTokenizer(s, SEPARATOR_CANDIDATE);
		ObjectFolder parent = mRootFolder;
		if (st.hasMoreElements()) {
			int countToken = st.countTokens();
			for (int i = 0; i < countToken - 1; i++) {
				String folder = st.nextToken();
				parent = getFolder(parent, folder);
			}

			Object o = parent.getNode(st.nextToken());
			if (o != null && !(o instanceof ObjectFolder)) {
				ObjectNode on = (ObjectNode)o;
				return on;
			}
		}
		return null;
	}

	/**
	 * ノードの追加
	 * @param s
	 * @param af
	 */
	public void addFile(String s, byte[] af) {
		StringTokenizer st = new StringTokenizer(s, SEPARATOR_CANDIDATE);
		ObjectFolder parent = mRootFolder;
		if (st.hasMoreElements()) {
			int countToken = st.countTokens();
			for (int i = 0; i < countToken - 1; i++) {
				String folder = st.nextToken();
				parent = getFolder(parent, folder);
			}
//			ObjectNode on;
			parent.addChild(/*on = */new ObjectNode(st.nextToken(), af != null ? Base64Encoder.encode(af) : null));
//			if (d.d.d) {
//				System.out.println(on.getName() + "," + on.getObject().toString());
//			}
		}
	}
	/**
	 * フォルダの追加
	 * @param s		フォルダパス
	 */
	public void addPath(String s) {
		StringTokenizer st = new StringTokenizer(s, SEPARATOR_CANDIDATE);
		ObjectFolder parent = mRootFolder;
		if (st.hasMoreElements()) {
			int countToken = st.countTokens();
			for (int i = 0; i < countToken; i++) {
				String folder = st.nextToken();
				parent = getFolder(parent, folder);
			}
		}
	}

	/**
	 * ファイルシステムの開放
	 * @param on
	 */
	public void releaseFileSystem() {
		release(mRootFolder);
		mRootFolder = null;
	}

	/**
	 * ファイルシステムのクローズ
	 * 全キャッシュの開放
	 *
	 * @param on
	 */
	private void release(ObjectNode on) {
		if (on instanceof ObjectFolder) {
			ObjectFolder of = (ObjectFolder)on;
			for (int i = 0; i < of.getChildren().size(); i++) {
				ObjectNode cn = of.getChildren().get(i);
				release(cn);
			}
			of.getChildren().clear();
		} else {
			on.setParent(null);
			on.setObject(null);
		}
	}

	/**
	 * フォルダ要素の取得
	 * 存在しなければ作成する。
	 * @param parent	親フォルダ
	 * @param tag		検索サブフォルダ名
	 * @param key		検索要素名
	 * @param value	検索要素値
	 * @return
	 */
	static ObjectFolder getFolder(ObjectFolder parent, String tag) {
		List<ObjectNode> v = parent.getNamedElement(tag);
		for (int i = 0; i < v.size(); i++) {
			ObjectNode on = v.get(i);
			if (on instanceof ObjectFolder)
				return (ObjectFolder) on;
		}
		ObjectFolder newf = new ObjectFolder(tag);
		parent.addChild(newf, true);
		return newf;
	}

	/**
	 * 転送用データの取得
	 * @return
	 */
	public byte[] getTransData() {
		try {
			AbstFile af = new AbstFile();
			PrintStream ps = new PrintStream(af.openOutputStream(""));
			XMLWriter.write(mRootFolder, "", ps);
			ps.close();
			return af.getData();
		} catch (Exception e) {
			log.error("XML作成エラー", e);
		}
		return null;
	}

	/**
	 * 転送用データの設定
	 * @param data
	 */
	public void setTransData(byte[] data) {
		XMLManager man = new XMLManager();
		AbstFile af = new AbstFile();
		af.setData(data);
		if (man.load(af)) {
			mRootFolder = man.getTree();
			decodeBinaryData(mRootFolder);
		}
	}

	/**
	 * 名前を持つフォルダのデータの取得
	 * @param parent
	 * @param s
	 * @return
	 */
	public byte[] searchNode(ObjectFolder parent, String s) {
		if (parent == null) {
			parent = mRootFolder;
		}
		for (int i = 0; i < parent.getChildren().size(); i++) {
			Object o = parent.getChildren().get(i);
			if (o instanceof ObjectFolder) {
				byte[] res = searchNode((ObjectFolder)o, s);
				if (res != null)
					return res;
			} else {
				ObjectNode on = (ObjectNode)o;
				if (on.getName() != null && on.getName().equalsIgnoreCase(s) && on.getObject() instanceof byte[]) {
					return (byte[])on.getObject();
				}
			}
		}
		return null;
	}

	/**
	 * 保持ノードすべてデコードする
	 * @param on
	 */
	public static void decodeBinaryData(ObjectNode on) {
		if (on instanceof ObjectFolder) {
			ObjectFolder of = (ObjectFolder)on;
			if (of.getChildren() != null) {
				for (int i = 0; i < of.getChildren().size(); i++) {
					ObjectNode on2 = of.getChildren().get(i);
					decodeBinaryData(on2);
				}
			}
		} else {
			Object data = on.getObject();
			if (data instanceof String) {
				on.setObject(Base64Decoder.decode((String)data));
			}
		}
	}

	/**
	 * 保持ノードすべてデコードする
	 * @param on
	 */
	public static void encodeBinaryData(ObjectNode on) {
		if (on instanceof ObjectFolder) {
			ObjectFolder of = (ObjectFolder)on;
			if (of.getChildren() != null) {
				for (int i = 0; i < of.getChildren().size(); i++) {
					ObjectNode on2 = of.getChildren().get(i);
					encodeBinaryData(on2);
				}
			}
		} else {
			Object data = on.getObject();
			if (data instanceof byte[]) {
				on.setObject(Base64Encoder.encode((byte[])data));
			}
		}
	}
}
