package jp.kitec.lib.util.tree;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;


/**
 * ツリーなどの階層構造における複数のノードを保持する親ノード。<BR>
 *
 * @since 　2002/12/20
 * @author　Kawae
 * @version 2002/12/20
 *
 * Copyright (c) 2002 KITec Inc,.. All rights reserved.
 */
public class ObjectFolder extends ObjectNode {

	/////////////////////////////////////////////////////////////////////////
	// フィールド
	private List<ObjectNode> mChildren;

	/**
	 * 指定された名前と親を持つ新たな親ノードを構築する。
	 *
	 * @param name		このノードの名前
	 * @param parent	このノードの親
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public ObjectFolder(String name) {
		super(name, null);
		mChildren = new ArrayList<ObjectNode>();
	}

	/**
	 * 階層構造内の全てのObjectFolderのノードを削除する
	 */
	public void removeAllElements() {
		for (int i = 0; i < mChildren.size(); i++) {
			Object o = mChildren.get(i);
			if (o instanceof ObjectFolder) {
				((ObjectFolder)o).removeAllElements();
			}
		}
		mChildren.clear();
	}

	/**
	 * 指定されたパスに対応する親ノード、およびノードからなる
	 * 階層構造を作成し、このノード下へ追加する。
	 *
	 * @param s	パス
	 * @return 最後に追加されたノード。
	 * 			ノードが追加されたなかった場合は、このノード。
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public ObjectFolder addFolderFromPath(String path) {
		ObjectFolder f = this;
		String[] st = path.split("[/\\\\]");

		for (String s : st) {
			if (s.isEmpty())
				continue;
			ObjectFolder cur = null;
			for (int i = 0; i < f.getChildren().size(); i++) {
				Object o = f.getChildren().get(i);
				if (o instanceof ObjectFolder) {
					ObjectFolder fol = (ObjectFolder)o;
					if (fol.getName().equalsIgnoreCase(s)) {
						cur = fol;
						break;
					}
				}
			}
			if (cur == null) {
				ObjectFolder of = new ObjectFolder(s);
				f.addChild(of);
				f = cur = of;
			} else
				f = cur;
		}
		return f;
	}

	/**
	 * このノードが保持する階層構造の直下の階層を対象として、指定された
	 * 名前を持つノード（親ノード含む）を検索し、該当するノードが存在
	 * する場合、そのノードを返す。
	 *
	 * @param name	ノードの名前
	 * @return ノード。該当するノードが存在しない場合は、null。
	 * @see #getNode(String, boolean)
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public ObjectNode getNode(String name) {
		return getNode(name, true);
	}

//	/**
//	 * 名前をもつ一番最初に見つかったフォルダを取得する。
//	 * @param name
//	 * @return
//	 */
//	public ObjectFolder getFolder(String name) {
//	    for (int i = 0; i < getChildlen().size(); i++) {
//	        Object o = getChildlen().elementAt(i);
//	        if (o instanceof ObjectFolder) {
//	            ObjectFolder of = (ObjectFolder)o;
//	            if (of.getName() != null && of.getName().equalsIgnoreCase(name))
//	                return of;
//	        }
//	    }
//	    return null;
//	}

	/**
	 * このノードが保持する階層構造の直下の階層を対象として、指定された
	 * 名前を持つノード、または親ノードを検索し、該当するノードが存在
	 * する場合、そのノードを返す。
	 *
	 * @param name				ノードの名前
	 * @param includeFolder	親ノードを検索対象とする場合、true。
	 * 							false を指定すると、名前が一致しても
	 * 							対象からは除外される。
	 * @return ノード。該当するノードが存在しない場合は、null。
	 * @see #getNode(String, boolean)
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public ObjectNode getNode(String name, boolean includeFolder) {
		for (int i = 0; i < mChildren.size(); i++) {
			ObjectNode o = mChildren.get(i);
			if (!includeFolder && o instanceof ObjectFolder)
				continue;
			ObjectNode on = o;
			if (on.getName().equalsIgnoreCase(name))
				return on;
		}
		return null;
	}

	/**
	 * このノードが保持する階層構造の全ノードを、１つの階層ごとに
	 * 親ノード、ノードの順に並び替える。
	 *
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public void sortFolder() {
		ArrayList<ObjectFolder> fold = new ArrayList<ObjectFolder>();
		ArrayList<ObjectNode> node = new ArrayList<ObjectNode>();

		for (int i = 0; i < mChildren.size(); i++) {
			ObjectNode o = mChildren.get(i);
			if (o instanceof ObjectFolder)
				fold.add((ObjectFolder)o);
			else
				node.add(o);
		}

		mChildren.clear();

		for (int i = 0; i < fold.size(); i++) {
			fold.get(i).sortFolder();
			mChildren.add(fold.get(i));
		}

		for (int i = 0; i < node.size(); i++)
			mChildren.add(node.get(i));
	}

	/**
	 * このノードが保持する階層構造のノード群（子ノード）を返す。
	 *
	 * @return ノード群
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public List<ObjectNode> getChildren() {
		return mChildren;
	}

	/**
	 * このノードの階層構造に、指定されたノードを追加する。
	 *
	 * @param n 新たなに追加するノード
	 * @see #addChild(ObjectNode, boolean) {
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public void addChild(ObjectNode n) {
		addChild(n, false);
//		if (mChildren.contains(n))
//			return;
//		for (int i = 0; i < mChildren.size(); i++) {
//			ObjectNode node = (ObjectNode)mChildren.elementAt(i);
//			if (n.getName().compareTo(node.getName()) == 0)
//				return;
//		}
//		mChildren.addElement(n);
	}

	/**
	 * このノードの階層構造に、指定されたノードを追加する。
	 *
	 * @param n			新たなに追加するノード
	 * @param sameregist	このノードと同じ名前を持つノードが存在する場合、
	 * 						階層構造に追加するか否かを指定する。
	 * 						追加する場合は、true。追加しない場合は、false。
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public void addChild(ObjectNode n, boolean sameregist) {
		if (mChildren.contains(n))
			return;
		if (!sameregist) {
			for (int i = 0; i < mChildren.size(); i++) {
				ObjectNode node = mChildren.get(i);
				if (n.getName().compareTo(node.getName()) == 0)
					return;
			}
		}
		n.setParent(this);
		mChildren.add(n);
	}

	/**
	 * このノードの階層構造の中から、指定された名前と同じ名前を持つ
	 * すべてのノードを返す。大文字小文字は区別しない。
	 *
	 * @param name ノードの名前
	 * @return 指定された名前と同じ名前を持つノード群
	 */
	public List<ObjectNode> getNamedElement(String s) {
		ArrayList<ObjectNode> res = new ArrayList<ObjectNode>();
		for (int i = 0; i < mChildren.size(); i++) {
			ObjectNode on = mChildren.get(i);
			if (on.getName().equalsIgnoreCase(s)) {
				res.add(on);
			}
		}
		return res;
	}

	/**
	 * このノードの階層構造のすべての階層を対象として、指定された名前と
	 * 同じ名前を持つノードを検索し、そのノードを返す。
	 * 同じ名前を持つノードが複数存在する場合、次の内容でノードが優先度
	 * が決まる。
	 * (1)所属する階層が異なる場合、階層が上のノード
	 * (2)同じ階層の場合、最初に見つかった（先に追加された）ノード
	 *
	 * @param name	ノードの名前
	 * @return 指定された名前と同じ名前を持つノード
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public ObjectNode findNode(String name) {
		LinkedList<ObjectFolder> queue = new LinkedList<ObjectFolder>();

		queue.add(this);
		while (!queue.isEmpty()) {
			ObjectFolder of = queue.removeFirst();
			for (int i = 0; i < of.mChildren.size(); i++) {
				ObjectNode on = of.mChildren.get(i);
				if (on instanceof ObjectFolder) {
					queue.add((ObjectFolder)on);
				} else {
					if (on.getName().equalsIgnoreCase(name)) {
						return on;
					}
				}
			}
		}
		return null;
	}

	public ObjectNode findNodeFirst(String name) {
		for (int i = 0; i < mChildren.size(); i++) {
			ObjectNode on = mChildren.get(i);
			if (on instanceof ObjectFolder) {
				ObjectNode tmp = ((ObjectFolder)on).findNode(name);
				if (tmp != null)
					return tmp;
			} else {
				if (on.getName().equalsIgnoreCase(name)) {
					return on;
				}
			}
		}
		return null;
	}

	/**
	 * ObjectFolderが持つ階層構造のすべてのノードを v に追加する。
	 *
	 * @param v	ノードを設定先となるコレクション
	 * @since   2010/04/12
	 * @author  matsuzaki
	 */
	public void getAllChildNode(Collection<? super ObjectNode> v) {
		for (int i = 0; i < getChildren().size(); i++) {
			Object o = getChildren().get(i);

			if (o instanceof ObjectFolder) {
				((ObjectFolder)o).getAllChildNode(v);
			} else if (o instanceof ObjectNode) {
				v.add((ObjectNode)o);
			}
		}
	}


	/**
	 * ObjectFolderにString値を設定
	 * valがnullの場合、追加しない
	 * @param of
	 * @param name
	 * @param val
	 * @author kawae
	 * @since 2004/11/05
	 */
	public void addNode(String name, String val) {
		if (val == null)
			return;
		addChild(new ObjectNode(name, val), true);
	}

	/**
	 * ObjectFolderにint値を設定
	 * @param of
	 * @param name
	 * @param val
	 * @author kawae
	 * @since 2004/11/05
	 */
	public void addNode(String name, int val) {
		addChild(new ObjectNode(name, Integer.toString(val)), true);
	}

	/**
	 * ObjectFolderにfloat値を設定
	 * @param of
	 * @param name
	 * @param val
	 * @author kawae
	 * @since 2004/11/05
	 */
	public void addNode(String name, float val) {
		addChild(new ObjectNode(name, Float.toString(val)), true);
	}

	/**
	 * ObjectFolderにdouble値を設定
	 * @param of
	 * @param name
	 * @param val
	 * @author kawae
	 * @since 2004/11/05
	 */
	public void addNode(String name, double val) {
		addChild(new ObjectNode(name, Double.toString(val)), true);
	}

	/**
	 * ObjectFolderにboolean値を設定
	 * @param of
	 * @param name
	 * @param val
	 * @author kawae
	 * @since 2004/11/05
	 */
	public void addNode(String name, boolean val) {
		addChild(new ObjectNode(name, val? "1" : "0"), true);
	}

	/**
	 * ObjectFolderにString値を設定
	 * valがnullの場合、追加しない
	 * @param of
	 * @param name
	 * @param val
	 * @author kawae
	 * @since 2004/11/05
	 */
	public static void addNodeString(ObjectFolder of, String name, String val) {
		if (val == null)
			return;
		of.addChild(new ObjectNode(name, val), true);
	}

	/**
	 * ObjectFolderにint値を設定
	 * @param of
	 * @param name
	 * @param val
	 * @author kawae
	 * @since 2004/11/05
	 */
	public static void addNodeInt(ObjectFolder of, String name, int val) {
		of.addChild(new ObjectNode(name, Integer.toString(val)), true);
	}

	/**
	 * ObjectFolderにfloat値を設定
	 * @param of
	 * @param name
	 * @param val
	 * @author kawae
	 * @since 2004/11/05
	 */
	public static void addNodeFloat(ObjectFolder of, String name, float val) {
		of.addChild(new ObjectNode(name, Float.toString(val)), true);
	}

	/**
	 * ObjectFolderにdouble値を設定
	 * @param of
	 * @param name
	 * @param val
	 * @author kawae
	 * @since 2004/11/05
	 */
	public static void addNodeDouble(ObjectFolder of, String name, double val) {
		of.addChild(new ObjectNode(name, Double.toString(val)), true);
	}

	/**
	 * ObjectFolderにboolean値を設定
	 * @param of
	 * @param name
	 * @param val
	 * @author kawae
	 * @since 2004/11/05
	 */
	public static void addNodeBool(ObjectFolder of, String name, boolean val) {
		of.addChild(new ObjectNode(name, val? "1" : "0"), true);
	}
}