/*
 * Copyright (c) 2010 KITec Inc,.. All rights reserved.
 * @author matsuzaki
 */
/**
 *
 */
package jp.kitec.kwt;

import jp.kitec.lib.geom.KLocalAxis;


/**
 * @author matsuzaki
 *
 */
public abstract class AbstractRichGraphics extends MinRichGraphics implements IRichGraphics {

	/**
	 * 1ドットの長さ
	 */
	protected double mDotPerLen;
	/**
	 * ビューポート
	 */
	private KViewPort mViewPort;
	/**
	 * デバイスサイズ
	 */
	protected KDimension mDimension;

	/**
	 * 1rad
	 */
	protected static final double RAD = Math.PI / 180;
	/** コサインの360度のテーブル */
	protected static final double COS_TABLE[] = new double[360];
	/** サインの360度のテーブル */
	protected static final double SIN_TABLE[] = new double[360];

	static {
		for (int s = 0; s < 360; s++) {
			COS_TABLE[s] = Math.cos((s) * RAD);
			SIN_TABLE[s] = Math.sin((s) * RAD);
		}
	}

	/**
	 *
	 */
	public AbstractRichGraphics() {
		mViewPort = new KViewPort();
		mDimension = new KDimension();
	}

	/**
	 * ビューポートの設定
	 * @param vp
	 */
	public void setViewPort(KViewPort vp) {
		mViewPort = vp;
	}

	/**
	 * ビューポートの取得
	 * @return
	 */
	public KViewPort getViewPort() {
		return mViewPort;
	}

	public void setSize(int w, int h) {
		mDimension.width = w;
		mDimension.height = h;
	}

	public KDimension getSize() {
		return mDimension.clone();
	}

	/**
	 * @return the dotPerLen
	 */
	@Override
	public double getDotPerLen() {
		return mDotPerLen;
	}

	/**
	 * ビューの設定
	 * @param x1
	 * @param y1
	 * @param x2
	 * @param y2
	 */
	public void setView(double x1, double y1, double x2, double y2) {
		setView(x1, y1, x2, y2, mDimension);
	}

	public void setView(double x1, double y1, double x2, double y2,
			KDimension d) {
		KViewPort vp = getViewPort();
		vp.setX1(x1 < x2 ? x1 : x2);
		vp.setX2(x1 < x2 ? x2 : x1);
		vp.setY1(y1 < y2 ? y1 : y2);
		vp.setY2(y1 < y2 ? y2 : y1);

		updateViewScale(d);
	}

	/**
	 * 画面中心の移動
	 * @param cx
	 * @param cy
	 */
	public void setViewCenter(double cx, double cy) {
		KViewPort vp = getViewPort();
		vp.setViewCenter(cx, cy);
		updateViewScale(mDimension);
	}

	/**
	 * 描画範囲を返します。
	 * @return
	 */
	@Override
	public KArea getDrawArea() {
		KArea a = new KArea();
		a.maximize();
		a.updateMinMax(devToRealX(0), devToRealY(0));
		a.updateMinMax(devToRealX(mDimension.width), devToRealY(mDimension.height));
		return a;
	}

	/**
	 * 実座標から画面座標へ（ｘ）
	 *
	 * @param	x	実座標値
	 * @return	デバイス座標値
	 */
	public int realToDevX(double x) {
		if (mDimension == null)
			return 0;
		return getViewPort().realToDevX(mDimension, x);
	}

	/**
	 * 実座標から画面座標へ（ｙ）
	 *
	 * @param	y	実座標値
	 * @return	デバイス座標値
	 */
	public int realToDevY(double y) {
		if (mDimension == null)
			return 0;
		return getViewPort().realToDevY(mDimension, y);
	}

	/**
	 * 実座標から画面座標へ（ｘ）
	 *
	 * @param	x	実座標値
	 * @return	デバイス座標値
	 */
	public double realToDevXd(double x) {
		if (mDimension == null)
			return 0;
		return getViewPort().realToDevXd(mDimension, x);
	}

	/**
	 * 実座標から画面座標へ（ｙ）
	 *
	 * @param	y	実座標値
	 * @return	デバイス座標値
	 */
	public double realToDevYd(double y) {
		if (mDimension == null)
			return 0;
		return getViewPort().realToDevYd(mDimension, y);
	}

	/**
	 * 画面座標から実座標へ（ｘ）
	 *
	 * @param	x	デバイス座標値
	 * @return		実座標値
	 */
	public double devToRealX(double x) {
		if (mDimension == null)
			return 0;
		return getViewPort().devToRealX(mDimension, x);
	}

	/**
	 * 画面座標から実座標へ（ｙ）
	 *
	 * @param	y	デバイス座標値
	 * @return		実座標値
	 */
	public double devToRealY(double y) {
		if (mDimension == null)
			return 0;
		return getViewPort().devToRealY(mDimension, y);
	}

	/**
	 * 1ドットの長さを更新
	 */
	public void updateViewScale(KDimension d) {
		if (d == null)
			return;
		mDotPerLen = getViewPort().dotPerLen(d);
	}

	public boolean isInCanvas(int x, int y) {
		if (x < 0 || x > mDimension.width)
			return false;
		if (y < 0 || y > mDimension.height)
			return false;
		return true;
	}

	protected double getx(double x, double y, KLocalAxis la) {
		double xx;
		if (la == null) {
			xx = realToDevXd(x);
		} else {
			xx = realToDevXd(la.getGlobalX(x, y));
		}
		return xx;
	}

	protected double gety(double x, double y, KLocalAxis la) {
		double yy;
		if (la == null) {
			yy = realToDevYd(y);
		} else {
			yy = realToDevYd(la.getGlobalY(x, y));
		}
		return yy;
	}
}
