/*
 * Copyright (c) 2009-2010 KITec Inc,.. All rights reserved.
 */
package jp.kitec.lib.io;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.StringTokenizer;

import jp.kitec.lib.util.StringUtil;

import org.apache.commons.lang.RandomStringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;




/**
 * 操作系のファイルユーティリティクラスです。
 *
 * @author $Author$
 * @version $Revision$ $Date::                           $
 */
public class FileUtil {

	//------------------------------------------------------------------
	//- fields
	//------------------------------------------------------------------

	/** ログ */
	private static final Log _log = LogFactory.getLog(FileUtil.class);



	//------------------------------------------------------------------
	//- constructors
	//------------------------------------------------------------------

	/**
	 * コンストラクタ（外部new禁止）
	 */
	protected FileUtil() {
	}



	//------------------------------------------------------------------
	//- methods
	//------------------------------------------------------------------

	/**
	 * ファイル（ディレクトリを含む）が存在するかチェックします。
	 *
	 * @param path ファイルへのパス
	 * @return ファイルが存在するならばtrueを返す。
	 */
	public static boolean exists(String path) {
		boolean result = false;
		try {
			File file = new File(path);
			result = file.exists();
		} catch (Exception e) {
			_log.error(e.toString());
		}
		return result;
	}

	/**
	 * 指定されたパス配下を全て削除します。<br/>
	 * 削除対象がシンボリックリンクの場合は、リンクだけ削除し配下ファイルは削除しません。<br/>
	 * <br/>
	 * 処理の流れは以下の通りです。<br/>
	 *   1.シンボリックリンクに対応するため、とりあえず削除してみる。<br/>
	 *     （シンボリックリンクなのにリンク先配下ファイルを削除しないための対策）<br/>
	 *   2.削除できたらファイルかシンボリックリンクなので、trueでメソッドを終了する。<br/>
	 *   3.削除できない場合はディレクトリなので、配下ファイルを削除するため再帰処理を行う。<br/>
	 *   4.配下ファイルを削除し終わったら自分自身を削除し、メソッドを終了する。<br/>
	 *
	 * @param file パス情報
	 * @return 全て削除できた場合はtrueを返す。
	 */
	public static boolean deleteAll(File file) {
		if (file.delete()) return true;

		File[] childFiles = file.listFiles();
		if (childFiles == null) return true;

		for (File childFile : childFiles) {
			deleteAll(childFile);
		}
		return file.delete();
	}

	/**
	 * ファイルの名称を取得します。<br/>
	 * パスを取り除いたファイルの名称のみを返します。<br/>
	 *
	 * @param path ファイルへのパス
	 * @return ファイルの名称
	 */
	public static String getFileName(String path) {
		if (path == null) return null;

		int lastIndex = path.lastIndexOf("\\");
		if (lastIndex > -1) return path.substring(lastIndex + 1);

		lastIndex = path.lastIndexOf("/");
		if (lastIndex > -1) return path.substring(lastIndex + 1);

		return path;
	}

	/**
	 * ファイル名から拡張子を取得します。
	 *
	 * @param fileName ファイル名
	 * @return ファイル拡張子
	 */
	public static String getFileExt(String fileName) {
		if (fileName == null) return null;
		int index = fileName.lastIndexOf('.');
		if (index == -1) return null;
		return fileName.substring(index + 1);
	}

	/**
	 * ファイル拡張子がリスト中に存在するかチェックします。
	 *
	 * @param extensions ファイル拡張子リスト
	 * @param target 存在チェック対象ファイル拡張子
	 * @return 存在する場合はtrueを返す。
	 */
	public static boolean isExtMatch(String extensions, String target) {
		StringTokenizer tokenizer = new StringTokenizer(extensions, " ");
		while (tokenizer.hasMoreTokens()) {
			String token = tokenizer.nextToken().toLowerCase();
			if (target.toLowerCase().equals(token)) return true;
		}
		return false;
	}

	/**
	 * ２つのパスを結合します。<br/>
	 * 区切り文字が２重になっている場合は１つにします。<br/>
	 * また、指定されたパスがnullの場合、空白が指定されたものとして扱います。<br/>
	 *
	 * @param path1 パス1
	 * @param path2 パス2
	 * @return 結合されたパス
	 */
	public static String concatPath(String path1, String path2) {
		if (path1 == null && path2 == null) return "";
		if (path1 == null || path1.equals("")) return path2;
		if (path2 == null || path2.equals("")) return path1;
		if (path1.equals("") && path2.equals("")) return "";

		String result = path1 + "/" + path2;
		while (result.indexOf("//") > -1) {
			result = result.replaceAll("//", "/");
		}
		while (result.indexOf("\\\\") > -1) {
			result = result.replaceAll("\\\\", "/");
		}
		return result;
	}

	/**
	 * 複数のパスを結合します。<br/>
	 * 区切り文字が２重になっている場合は１つにします。<br/>
	 * また、指定されたパスがnullの場合、空白が指定されたものとして扱います。<br/>
	 *
	 * @param paths パス配列
	 * @return 結合されたパス
	 */
	public static String concatPath(String[] paths) {
		String result = null;

		for (String path : paths) {
			result = concatPath(result, path);
		}

		return result;
	}

	/**
	 * 指定ディレクトリ配下にファイルが存在するかチェックします。
	 *
	 * @param rootFile パス情報
	 * @return ファイル数
	 */
	public static int findFile(File rootFile) {
		int count = 0;
		if (rootFile.isDirectory()) {
			File[] files = rootFile.listFiles();
			for (File file : files) {
				count += findFile(file);
			}
		} else {
			count++;
		}
		return count;
	}

	/**
	 * 指定された接頭辞をディレクトリ名の生成に使用して、デフォルトの一時ファイルディレクトリにディレクトリを作成します。<br/>
	 * このメソッドは、createTempDir(prefix, null) と同じです。<br/>
	 *
	 * @param prefix ファイル名を生成するために使用される接頭辞文字列。3 文字以上の長さが必要。
	 * @throws IllegalArgumentException 引数が3文字に満たない場合。
	 */
	public static File createTempDir(String prefix) {
		return createTempDir(prefix, null);
	}

	/**
	 * 指定された接頭辞をディレクトリ名の生成に使用して、デフォルトの一時ファイルディレクトリにディレクトリを作成します。
	 *
	 * @param prefix ファイル名を生成するために使用される接頭辞文字列。3 文字以上の長さが必要。
	 * @param dir ファイルが生成されるディレクトリ。nullを指定した場合は、デフォルトの一時ファイルディレクトリが使用されます。
	 * @throws IllegalArgumentException 引数が3文字に満たない場合。
	 */
	public static File createTempDir(String prefix, File dir) {
		if (prefix == null || prefix.length() < 3)
			throw new IllegalArgumentException("length of the prefix is not 3 character or more.");

		String baseDirPath = System.getProperty("java.io.tmpdir");
		if (dir != null) {
			try {
				baseDirPath = dir.getCanonicalPath();
			} catch (IOException e) {
			}
		}
		if (!baseDirPath.endsWith(File.separator)) baseDirPath += File.separator;

		String newTmpDirPath = null;
		while (true) {
			newTmpDirPath = baseDirPath + prefix + RandomStringUtils.randomAlphanumeric(8);
_log.debug("newTmpDirPath[" + newTmpDirPath + "]");
			if (!(new File(newTmpDirPath)).exists()) break;
		}

		File newTmpDir = new File(newTmpDirPath);
		return (newTmpDir.mkdirs()? newTmpDir: null);
	}

	/**
	 * ファイルをコピーします。
	 *
	 * @param fileFrom ファイルコピー元
	 * @param fileTo ファイルコピー先
	 */
	public static boolean copy(File fileFrom, File fileTo) {
		if (!fileFrom.isFile()) return false;

		boolean result = false;
		InputStream inputStream = null;

		try {
			inputStream = new BufferedInputStream(new FileInputStream(fileFrom));
			FileIOUtil.write(fileTo, inputStream);

			result = true;
		} catch (Exception e) {
			_log.error("", e);
		} finally {
			FileIOUtil.close(inputStream);
		}

		return result;
	}

	/**
	 * ディレクトリをコピーします。<br/>
	 * FileToが存在する場合は、その配下にFileFrom配下(FileFrom自身も含む)のディレクトリやファイルをコピーします。<br/>
	 * FileToが存在しない場合は、FileToを作成し、その配下にFileFrom配下(FileFrom自身は含まない)のディレクトリやファイルをコピーします。<br/>
	 *
	 * @param fileFrom ディレクトリコピー元
	 * @param fileTo ディレクトリコピー先
	 * @throws FileNotFoundRuntimeException Fromディレクトリが見つからないかファイルだった場合、Toディレクトリがファイルだった場合にスローします。
	 * @throws DirectoryNotMadeRuntimeException ディレクトリが作成できなかった場合にスローします。
	 */
	public static boolean copyDir(File fileFrom, File fileTo) {
		if (!fileFrom.exists()) throw new FileNotFoundRuntimeException(fileFrom);
		if (!fileFrom.isDirectory()) throw new FileNotFoundRuntimeException(fileFrom);
		if (fileTo.isFile()) throw new FileNotFoundRuntimeException(fileTo);

		File newDir = null;
		if (fileTo.exists()) {
			newDir = new File(fileTo, fileFrom.getName());
		} else {
			newDir = fileTo;
		}
		if (!newDir.mkdirs()) throw new DirectoryNotMadeRuntimeException(newDir);

		boolean result = true;
		File[] childFiles = fileFrom.listFiles();
		for (File childFile : childFiles) {
			if (childFile.isDirectory()) {
				result = copyDir(childFile, newDir);
			} else {
				File newFile = new File(newDir, childFile.getName());
				result = copy(childFile, newFile);
			}
			if (!result) break;
		}

		return result;
	}

	/**
	 * 数値から階層構造ディレクトリ名を生成します。<br/>
	 * 10203, 6, 2 ----> /01/0102/010203<br/>
	 * 1020304, 9, 3 --> /001/001020/001020304<br/>
	 *
	 * @param num 対象数値
	 * @param max 最大桁数
	 * @param span 分割間隔
	 * @return 階層ディレクトリ名
	 */
	public static String createHierarchyDirName(long num, int max, int span) {
		if (max < span) max = span;
		if (max % span != 0) max = span - (max % span);

		String paddedNum = StringUtil.leftPad(num, max, '0');
		StringBuilder buf = new StringBuilder();
		for (int i = span; i <= max; i += span) {
			String splitedNum = paddedNum.substring(0, i);
			buf.append('/');
			buf.append(splitedNum);
		}

		return buf.toString();
	}



} // end-class
