package jp.kitec.lib.util;


/**
 * 文字列の解析に対して、複数のトークン区切り文字を
 * 使用できるトークン解析クラス。
 * 例）	区切り文字：xyz
 * 		解析文字列：aaxbyzccx
 * 		分割される文字列：
 * 					aa
 * 					b
 * 					(空文字)
 * 					cc
 * 					(空文字)
 *
 * @since 　2003/02/13
 * @author　Kawae
 * @version 2003/02/13
 *
 * Copyright (c) 2002 KITec Inc,.. All rights reserved.
 */
public class StringToken2 {

	/////////////////////////////////////////////////////////////////////////
	// フィールド

	/** 解析文字列 */
	private String str;
	/** 区切り文字列 */
	private String deli;
	/** 要素数 */
	private int count;
	/** 区切り文字列の個々の区切り文字 */
	private String[] delis;


	/**
	 * 指定された文字列に対する StringToken2 を作成する。
	 * delim は、トークンを分ける区切り文字。区切り文字そのものは
	 * トークンとして扱われない。
	 *
	 * @param s	解析される文字列
	 * @param del	区切り文字
	 * @since   2003/02/13
	 * @author  Kawae
	 * @version 2003/02/13
	 */
	public  StringToken2(String s, String del) {
		str = s;
		setDelimitor(del);
		updateCountTokens();
	}

	/**
	 * トークンの数を更新する。
	 */
	private void updateCountTokens() {
		String tmp = str;
		count = 0;
		while (hasMoreElements()) {
			nextToken();
			count ++;
		}
		str = tmp;
	}

	/**
	 * 空の StringToken2 を作成する。
	 *
	 * @since   2003/02/13
	 * @author  Kawae
	 * @version 2003/02/13
	 */
	public StringToken2() {
	}

	/**
	 * 指定された文字列に含まれるトークンの数を返す。
	 * java.util.StringTokenizerとは挙動が違うので注意。
	 * @return
	 */
	public int countTokens() {
		return count;
	}

	/**
	 * 新たに解析する文字列を設定する。
	 *
	 * @param s	解析される文字列
	 * @since   2003/02/13
	 * @author  Kawae
	 * @version 2003/02/13
	 */
	public void setText(String s) {
		str = s;
		updateCountTokens();
	}

	/**
	 * 新たな区切り文字を設定する。
	 * 区切り文字そのものは、トークンとして扱われない。
	 *
	 * @param s	区切り文字
	 * @since   2003/02/13
	 * @author  Kawae
	 * @version 2003/02/13
	 */
	public void setDelimitor(String s) {
		deli = s;
		delis = new String[deli.length()];
		for (int i = 0; i < deli.length(); i++)
			delis[i] = deli.substring(i, i + 1);
		updateCountTokens();
	}

	/**
	 * トークナイザの文字列で利用できるトークンが
	 * まだあるかどうかを判定する。
	 *
	 * @return	文字列内の現在の位置の後ろに 1 つ以上の
	 * 			トークンがある場合だけ true、そうでない
	 * 			場合は false
	 * @since   2003/02/13
	 * @author  Kawae
	 * @version 2003/02/13
	 */
	public boolean hasMoreElements() {
		return str != null;
	}

	/**
	 * 文字列トークナイザから次のトークンを返す。
	 *
	 * @return	文字列トークナイザからの次のトークン
	 * @since   2003/02/13
	 * @author  Kawae
	 * @version 2003/02/13
	 */
	public String nextToken() {
		String res = null;
		if (str != null) {
			int min = -1;
			int tmp;
			for (String deli2 : delis) {
				if ((tmp = str.indexOf(deli2)) >= 0) {
					if (min < 0)
						min = tmp;
					else if (tmp < min)
						min = tmp;
				}
			}
			if (min >= 0) {
				res = str.substring(0, min);
				str = str.substring(min + 1);
			} else if (str.length() >= 0) {
				res = str;
				str = null;
			}
		}
		return res;
	}
}