package jp.kitec.lib.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import jp.kitec.lib.io.FileIOUtil;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;


/**
 * オブジェクトの圧縮、解凍を行う。<BR>
 *
 * @since 　2002/12/20
 * @author　Kawae
 * @version 2002/12/20
 *
 * Copyright (c) 2002 KITec Inc,.. All rights reserved.
 */
public class CompressData {

	private static final Log log = LogFactory.getLog(CompressData.class);

	/**
	 * 空の CompressData を構築する。
	 *
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public CompressData() {
	}


	/**
	 * 指定されたバイト配列を圧縮し、返す。
	 *
	 * @param o 圧縮するバイト配列
	 * @return	バイトデータ形式の圧縮されたオブジェクト
	 * @see #decompress(Object)
	 * @since   2015/11/12
	 */
	public static byte[] compress(byte[] o) {
		if (o == null)
			return null;
		try {
			ByteArrayOutputStream baus = new ByteArrayOutputStream();
			GZIPOutputStream zos = new GZIPOutputStream(baus);
			zos.write(o);
			zos.close();
			baus.close();
			return baus.toByteArray();
		} catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * 指定されたオブジェクトを圧縮し、返す。
	 *
	 * @param o 圧縮するオブジェクト
	 * @return	バイトデータ形式の圧縮されたオブジェクト
	 * @see #decompress(Object)
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public static byte[] compressUnsafe(Object o) {

		if (o == null)
			return null;
		try {
			ByteArrayOutputStream baus = new ByteArrayOutputStream();
			GZIPOutputStream zos = new GZIPOutputStream(baus);
			ObjectOutputStream oos = new ObjectOutputStream(zos);
			oos.writeObject(o);
			oos.close();
			zos.close();
			baus.close();
			return baus.toByteArray();
		} catch (Exception e) {
			log.error("圧縮エラー", e);
		}
		return null;
	}


	/**
	 * 指定されたバイト配列を解凍し、返す。
	 *
	 * @param o 解凍するバイト配列
	 * @return	解凍されたバイト配列
	 * @see #compress(Object)
	 * @since   2015/11/12
	 */
	public static byte[] decompress(byte[] b) {
		if (b == null)
			return null;

		try {
			ByteArrayInputStream baus = new ByteArrayInputStream(b);
			GZIPInputStream zos = new GZIPInputStream(baus);
			byte[] res = FileIOUtil.readBytes(zos);
			zos.close();
			baus.close();
			return res;
		} catch (IOException e) {
			e.printStackTrace();
			return b;
		} catch (Exception e) {
			e.printStackTrace();
		}

		return null;
	}

	/**
	 * 指定されたオブジェクトを解凍し、返す。
	 *
	 * @param o 解凍するオブジェクト。byte[] のみが処理対象となる。
	 * @return	解凍されたオブジェクト
	 * @see #compress(Object)
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public static Object decompressUnsafe(Object o) {

		if (o == null || !(o instanceof byte[]))
			return o;

		byte[] b = (byte[])o;
		try {
			ByteArrayInputStream baus = new ByteArrayInputStream(b);
			GZIPInputStream zos = new GZIPInputStream(baus);
			ObjectInputStream ois = new ObjectInputStream(zos);
			Object res = ois.readObject();
			ois.close();
			zos.close();
			baus.close();
			return res;
		} catch (IOException e) {
			log.error("展開エラー", e);
			return o;
		} catch (Exception e) {
			log.error("展開エラー", e);
		}

		return null;
	}
}