/*
 * Copyright (c) 2010 KITec Inc,.. All rights reserved.
 */
package jp.kitec.lib.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;



/**
 * Enumユーティリティクラス
 *
 * @author $Author$
 * @version $Revision$ $Date::                           $
 */
public class EnumUtil {

	//------------------------------------------------------------------
	//- constructors
	//------------------------------------------------------------------

	/**
	 * コンストラクタ（外部new禁止）
	 */
	protected EnumUtil() {
	}



	//------------------------------------------------------------------
	//- methods
	//------------------------------------------------------------------

	/**
	 * 前のEnumを取得します。
	 *
	 * @return 前のEnum
	 */
	@SuppressWarnings("unchecked")
	public static <T extends Enum> T prev(T targetEnum) {
		int max = targetEnum.getClass().getEnumConstants().length;
		int now = targetEnum.ordinal();
		int prev = (now == 0)? max - 1: now - 1;

		return (T)targetEnum.getClass().getEnumConstants()[prev];
	}

	/**
	 * 次のEnumを取得します。
	 *
	 * @return 次のEnum
	 */
	@SuppressWarnings("unchecked")
	public static <T extends Enum> T next(T targetEnum) {
		int max = targetEnum.getClass().getEnumConstants().length;
		int now = targetEnum.ordinal();
		int next = (now == max - 1)? 0: now + 1;

		return (T)targetEnum.getClass().getEnumConstants()[next];
	}

	/**
	 * IDからEnumを取得します。
	 *
	 * @param enumType EnumWithIdを実装したEnum
	 * @param id ID
	 * @return EnumWithId拡張Enum
	 */
	public static <T extends Enum<?> & EnumWithId> T getById(Class<T> enumType, Integer id) {
		T result = null;
		for (T e: enumType.getEnumConstants()) {
			if (e.getId().equals(id)) {
				result = e;
				break;
			}
		}
		return result;
	}

	/**
	 * 名称からEnumを取得します。
	 *
	 * @param enumType EnumWithNameを実装したEnum
	 * @param name 名称
	 * @return EnumWithName拡張Enum
	 */
	public static <T extends Enum<?> & EnumWithName> T getByName(Class<T> enumType, String name) {
		T result = null;
		for (T e: enumType.getEnumConstants()) {
			if (e.getName().equals(name)) {
				result = e;
				break;
			}
		}
		return result;
	}

	/**
	 * EnumWithIdに定義してあるIDをIntegerコレクションで取得します。
	 *
	 * @param enumType EnumWithIdを実装したEnum
	 * @return IDコレクション
	 */
	public static <T extends Enum<?> & EnumWithId> Collection<Integer> getIds(Class<T> enumType) {
		T[] es = enumType.getEnumConstants();
		List<Integer> result = new ArrayList<Integer>();
		for (T e: es) result.add(e.getId());

		return result;
	}

	/**
	 * EnumWithNameに定義してある名称をStringコレクションで取得します。
	 *
	 * @param enumType EnumWithNameを実装したEnum
	 * @return 名称コレクション
	 */
	public static <T extends Enum<?> & EnumWithName> Collection<String> getNames(Class<T> enumType) {
		T[] es = enumType.getEnumConstants();
		List<String> result = new ArrayList<String>();
		for (T e: es) result.add(e.getName());

		return result;
	}

	/**
	 * EnumWithIdに定義してあるIDをInteger配列で取得します。
	 *
	 * @param enumType EnumWithIdを実装したEnum
	 * @return ID配列
	 */
	public static <T extends Enum<?> & EnumWithId> Integer[] getIdArray(Class<T> enumType) {
		T[] es = enumType.getEnumConstants();
		Integer[] result = new Integer[es.length];
		for (int i = 0; i < es.length; i++) result[i] = es[i].getId();

		return result;
	}

	/**
	 * EnumWithNameに定義してある名称をString配列で取得します。
	 *
	 * @param enumType EnumWithNameを実装したEnum
	 * @return 名称配列
	 */
	public static <T extends Enum<?> & EnumWithName> String[] getNameArray(Class<T> enumType) {
		T[] es = enumType.getEnumConstants();
		String[] result = new String[es.length];
		for (int i = 0; i < es.length; i++) result[i] = es[i].getName();

		return result;
	}



} // end-enum

