/**
 * Copyright (c) 2009 KITec Inc,.. All rights reserved.
 */
package jp.kitec.lib.util;

import java.text.DecimalFormat;
import java.util.Collection;



/**
 * 数値ユーティリティクラス
 *
 * @author $Author$
 * @version $Revision$ $Date::                           $
 */
public class NumberUtil {

	//------------------------------------------------------------------
	//- constructors
	//------------------------------------------------------------------

	/**
	 * コンストラクタ（外部new禁止）
	 */
	private NumberUtil() {
	}



	//------------------------------------------------------------------
	//- methods
	//------------------------------------------------------------------

	/**
	 * 数字に３桁区切りで','を挿入します。(123,456,789)
	 *
	 * @param value 数字
	 * @return カンマ区切り数字
	 */
	public static String formComma(int value) {
		return formComma(Integer.toString(value));
	}
	public static String formComma(long value) {
		return formComma(Long.toString(value));
	}
	public static String formComma(float value) {
		return formComma(Float.toString(value));
	}
	public static String formComma(double value) {
		return formComma(Double.toString(value));
	}
	public static String formComma(String value) {
		String result = "";
		DecimalFormat df = null;

		if (value.indexOf(".") == -1) {
			df = new DecimalFormat(",##0");
			result = df.format(Long.parseLong(value));
		} else {
			df = new DecimalFormat(",##0.00");
			result = df.format(Double.parseDouble(value));
		}

		return result;
	}

	/**
	 * Yen書式に整形します。(\123,456,789)
	 *
	 * @param value 数字
	 * @return Yen書式数字
	 */
	public static String formYen(int value) {
		return formYen(Integer.toString(value));
	}
	public static String formYen(long value) {
		return formYen(Long.toString(value));
	}
	public static String formYen(String value) {
		DecimalFormat df = null;

		if (value.indexOf(".") == -1)
			df = new DecimalFormat("\\,##0");
		else
			df = new DecimalFormat("\\,##0.00");

		return df.format(Float.parseFloat(value));
	}

	/**
	 * 整数か検査します<br>
	 * ※注意！　intかどうかはチェックしません。long等でも可。Intergerは「整数」の意味です。
	 *
	 * @param value 値
	 * @return 整数ならtrue。違うならfalseを返します。
	 */
	public static boolean isInteger(String value) {
		boolean result = true;

		try {
			Long.parseLong(value);
		} catch (NumberFormatException ex) {
			result = false;
		}

		return result;
	}

	/**
	 * 小数かどうか検査します（整数でも可）
	 *
	 * @param value 値
	 * @return 小数（整数）ならtrue。違うならfalseを返します。
	 */
	public static boolean isDecimal(String value) {
		boolean result = true;

		try {
			Double.parseDouble(value);
		} catch (NumberFormatException ex) {
			result = false;
		}

		return result;
	}

	/**
	 * 小数かどうか検査します。
	 *
	 * @param value 値
	 * @return 小数ならtrue。違うならfalseを返します。
	 */
	public static boolean isDecimal(double value) {
		return value != Math.round(value);
	}

	/**
	 * 四捨五入した値を求めます。
	 *
	 * @param value 値
	 * @param position 位。小数第1位は-1、第2位は-2で表す。
	 * @return 四捨五入された値
	 */
	public static double round(double value, int position) {
		double result = 0;

		long invokution = 1;
		if (position > 0) {
			invokution = calcInvokution(10, position);
		} else if (position < -1) {
			invokution = calcInvokution(10, Math.abs(position) - 1);
		}

		if (position > 0) {
			double invokutionValue = value / invokution;
			double roundValue = Math.round(invokutionValue);
			result = roundValue * invokution;
		} else if (position < 0) {
			double invokutionValue = value * invokution;
			double roundValue = Math.round(invokutionValue);
			result = roundValue / invokution;
		}
		return result;
	}

	/**
	 * 累乗を求めます。
	 *
	 * @param value 値
	 * @param num 掛ける回数
	 * @return value と num の累乗
	 */
	public static long calcInvokution(int value, int num) {
		long result = value;
		for(int i = 0; i < num - 1; i++) {
			result = result * value;
		}
		return result;
	}

	/**
	 * 対象の文字列をIntegerに変換します。<br/>
	 * 変換できない場合、デフォルトのIntegerを返します。<br/>
	 *
	 * @param value 変換処理対象文字列
	 * @param defaultInteger デフォルトのInteger
	 * @return 変換後のInteger。変換できない場合はデフォルトのInteger
	 */
	public static Integer parseInteger(String value, Integer defaultInteger) {
		Integer result = null;
		try {
			result = new Integer(value);
		} catch (NumberFormatException nfe) {
			result = defaultInteger;
		}
		return result;
	}

	/**
	 * 対象の文字列をIntegerに変換します。<br/>
	 * 変換できない場合、デフォルト値のIntegerを返します。<br/>
	 *
	 * @param value 変換処理対象文字列
	 * @param defaultInt デフォルト値のInteger
	 * @return 変換後のInteger。変換できない場合はデフォルト値のInteger
	 */
	public static Integer parseInteger(String value, int defaultInt) {
		return parseInteger(value, Integer.valueOf(defaultInt));
	}

	/**
	 * 対象のIntegerを文字列に変換します。<br/>
	 * 対象がnullの場合は、defaultStringの値を返します。<br/>
	 *
	 * @param value 変換対象Integer
	 * @param defaultString nullの場合に返す値
	 * @return 変換対象の文字列表現。変換対象がnullの場合はdefaultString
	 */
	public static String toString(Integer value, String defaultString) {
		return (value == null)? defaultString: value.toString();
	}

	/**
	 * booleanをintに変換します。<br/>
	 * trueは1に、falseは0にします。<br/>
	 *
	 * @param value boolean値
	 * @return 変換後の値
	 */
	public static int toInt(boolean value) {
		return value? 1: 0;
	}

	public static double max(Collection<Number> numbers) {
		double result = Double.MIN_VALUE;
		for (Number number: numbers) {
			result = Math.max(number.doubleValue(), result);
		}
		return result;
	}

	public static float max(float[] floats) {
		float result = Float.MIN_VALUE;
		for (float number: floats) {
			result = Math.max(number, result);
		}
		return result;
	}

	public static double max(double[] doubles) {
		double result = Double.MIN_VALUE;
		for (double number: doubles) {
			result = Math.max(number, result);
		}
		return result;
	}



} // end-class

