/*
 * Copyright (c) 2009 KITec Inc,.. All rights reserved.
 */
package jp.kitec.lib.util;

import java.text.CharacterIterator;
import java.text.StringCharacterIterator;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;



/**
 * 文字変換クラス
 *
 * @author $Author$
 * @version $Revision$ $Date::                           $
 */
public class CharConverter {

	//------------------------------------------------------------------
	//- constants
	//------------------------------------------------------------------

	/** 半角整数チェック用文字列 */
	public final static String HAN_NUM =
		"1234567890";

	/** 半角大英字チェック用文字列 */
	public final static String HAN_CAPITAL_ALPHABET =
		"ABCDEFGHIJKLMNOPQRSTUVWXYZ";

	/** 半角小英字チェック用文字列 */
	public final static String HAN_SMALL_ALPHABET =
		"abcdefghijklmnopqrstuvwxyz";

	/** 半角カタカナチェック用文字列 */
	public final static String HAN_KATA =
		"ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐﾑﾒﾓﾔﾕﾖﾗﾘﾙﾚﾛﾜｦﾝ" +
		"ｧｨｩｪｫｯｬｭｮｰ";

	/** 全角整数チェック用文字列 */
	public final static String ZEN_NUM =
		"１２３４５６７８９０";

	/** 全角大英字チェック用文字列 */
	public final static String ZEN_CAPITAL_ALPHABET =
		"ＡＢＣＤＥＦＧＨＩＪＫＬＭＮＯＰＱＲＳＴＵＶＷＸＹＺ";

	/** 全角小英字チェック用文字列 */
	public final static String ZEN_SMALL_ALPHABET =
		"ａｂｃｄｅｆｇｈｉｊｋｌｍｎｏｐｑｒｓｔｕｖｗｘｙｚ";

	/** 全角カタカナチェック用文字列 */
	public final static String ZEN_KATA =
		"アイウエオカキクケコサシスセソタチツテトナニヌネノハヒフヘホマミムメモヤユヨラリルレロワヲン" +
		"ァィゥェォッャュョー";

	/** 全角カタカナチェック用文字列 */
	public final static String ZEN_HIRA =
		"あいうえおかきくけこさしすせそたちつてとなにぬねのはひふへほまみむめもやゆよらりるれろわをん";

	/** 半角濁点カタカナチェック用文字列 */
	public final static String HAN_DAKUKATA =
		"ｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾊﾋﾌﾍﾎ";

	/** 全角濁点カタカナチェック用文字列 */
	public final static String ZEN_DAKUKATA =
		"ガギグゲゴザジズゼゾダヂヅデドバビブベボ";

	/** 半角半濁点カタカナチェック用文字列 */
	public final static String HAN_HANDAKUKATA =
		"ﾊﾋﾌﾍﾎ";

	/** 全角半濁点カタカナチェック用文字列 */
	public final static String ZEN_HANDAKUKATA =
		"パピプペポ";



	//------------------------------------------------------------------
	//- fields
	//------------------------------------------------------------------

	/** ログ */
	private static Log _log = LogFactory.getLog(CharConverter.class);



	//------------------------------------------------------------------
	//- constructors
	//------------------------------------------------------------------

	/**
	 * コンストラクター
	 */
	public CharConverter() {
		super();
	}



	//------------------------------------------------------------------
	//- methods
	//------------------------------------------------------------------

	/**
	 * 指定文字列中の全角を半角に変換する
	 * @param src 変換対象文字列
	 * @return 変換された文字列
	 */
	public static String convertZenToHan(String target) {
		return convertCharacter(target,
			ZEN_NUM + ZEN_CAPITAL_ALPHABET + ZEN_SMALL_ALPHABET + ZEN_KATA + ZEN_HIRA + "　",
			HAN_NUM + HAN_CAPITAL_ALPHABET + HAN_SMALL_ALPHABET + HAN_KATA + HAN_KATA + " ");
	}

	/**
	 * 指定文字列中の半角を全角に変換する
	 * @param src 変換対象文字列
	 * @return 変換された文字列
	 */
	public static String convertHanToZen(String src) {
		StringBuffer result = new StringBuffer();
		char[] srcArray = src.toCharArray();

		char keepChar = 0;
		for (int i = 0; i <= srcArray.length; i++) {
			char srcChar = (i == srcArray.length)? 0: srcArray[i];
			String dstChar = null;

			if (keepChar != 0) {
				StringBuffer buf = new StringBuffer();
				buf.append(keepChar);

				if (srcChar == 'ﾞ') {
					dstChar = convertHanToZenDakuten(buf.toString());
				} else if (srcChar == 'ﾟ') {
					dstChar = convertHanToZenHandakuten(buf.toString());
				} else {
					dstChar = convertHanToZenNormal(buf.toString());
				}
				result.append(dstChar);

				if (srcChar == 'ﾞ' || srcChar == 'ﾟ') {
					keepChar = 0;
					continue;
				}
			}

			keepChar = srcChar;
		}

		return result.toString();
	}

	/**
	 * 指定文字列中の半角カタカナを全角カタカナに変換する
	 * @param src 変換対象文字列
	 * @return 変換された文字列
	 */
	public static String convertHanKataToZenKata(String src) {
		StringBuffer result = new StringBuffer();
		char[] srcArray = src.toCharArray();

		char keepChar = 0;
		for (int i = 0; i <= srcArray.length; i++) {
			char srcChar = (i == srcArray.length)? 0: srcArray[i];
			String dstChar = null;

			if (keepChar != 0) {
				StringBuffer buf = new StringBuffer();
				buf.append(keepChar);

				if (srcChar == 'ﾞ') {
					dstChar = convertHanToZenDakuten(buf.toString());
				} else if (srcChar == 'ﾟ') {
					dstChar = convertHanToZenHandakuten(buf.toString());
				} else {
					dstChar = convertCharacter(buf.toString(), HAN_KATA, ZEN_KATA);
				}
				result.append(dstChar);

				if (srcChar == 'ﾞ' || srcChar == 'ﾟ') {
					keepChar = 0;
					continue;
				}
			}

			keepChar = srcChar;
		}

		return result.toString();
	}

	/**
	 * 指定文字列中の半角を全角に変換する（通常文字が対象。濁点、半濁点は変換しない）
	 * @param src 変換対象文字列
	 * @return 変換された文字列
	 */
	private static String convertHanToZenNormal(String src) {
		return convertCharacter(src,
			HAN_NUM + HAN_CAPITAL_ALPHABET + HAN_SMALL_ALPHABET + HAN_KATA,
			ZEN_NUM + ZEN_CAPITAL_ALPHABET + ZEN_SMALL_ALPHABET + ZEN_KATA);
	}

	/**
	 * 指定文字列中の半角を全角に変換する（濁点文字が対象）
	 * @param src 変換対象文字列
	 * @return 変換された文字列
	 */
	private static String convertHanToZenDakuten(String src) {
		return convertCharacter(src,
			HAN_DAKUKATA,
			ZEN_DAKUKATA);
	}

	/**
	 * 指定文字列中の半角を全角に変換する（半濁点文字が対象）
	 * @param src 変換対象文字列
	 * @return 変換された文字列
	 */
	private static String convertHanToZenHandakuten(String src) {
		return convertCharacter(src,
			HAN_HANDAKUKATA,
			ZEN_HANDAKUKATA);
	}

	/**
	 * 文字列を変換する
	 * @param target チェック対象文字列
	 * @param before 変換対象の文字リスト
	 * @param after 変換後の文字リスト
	 * @return 変換された文字列
	 */
	public static String convertCharacter(String target, String before, String after) {
		if (target == null || target.compareTo("") == 0) return target;

		StringBuffer result = new StringBuffer();
		StringCharacterIterator tar = new StringCharacterIterator(target);
		char tarChar;
		for (tarChar = tar.first(); tarChar != CharacterIterator.DONE; tarChar = tar.next()) {

			int i;
			for(i = 0; i < before.length(); i++) {
				if(tarChar == before.charAt(i)) {
					result.append(after.charAt(i));
_log.debug("tarChar[" + tarChar + "]");
_log.debug("beforeChar[" + before.charAt(i) + "]");
_log.debug("afterChar[" + after.charAt(i) + "]");
					break;
				}
			}
			if (i == before.length()) {
				result.append(tarChar);
			}
		}
		return result.toString();
	}



} // end-class
