package jp.kitec.lib.io;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

import jp.kitec.lib.util.tree.ObjectFolder;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;


/**
 * @author kawae
 */
public class FileSystemUtil {

	private static final Log log = LogFactory.getLog(FileSystemUtil.class);

	/**
	 * ファイルシステムへファイルの追加
	 * @param fs
	 * @param baseDir		基本フォルダの場所。ファイルはすべてこのフォルダの相対パスとして管理される
	 * @param targetPath	対象の基本フォルダからのファイルパス
	 * @param cachefile		ファイルをキャッシュするか
	 */
	public static void addFileSystem(FileSystem fs, String baseDir, String targetPath, boolean cachefile, String[] exts) {
		if (baseDir.indexOf('\\') >= 0)
			baseDir = baseDir.replace('\\', FileSystem.SEPARATOR);

		if (targetPath.indexOf("\\") >= 0)
			targetPath = targetPath.replace('\\', FileSystem.SEPARATOR);
		//フォルダ下モードの場合
		if (targetPath.equalsIgnoreCase("*")) {
			File f =  new File(baseDir);
			if (f.isDirectory()) {
				String[] files = f.list();
				for (int i = 0; i < files.length; i++) {
					addFileSystem(fs, baseDir, files[i], fs.mRootFolder, cachefile, exts);
				}
			}
		} else {
			addFileSystem(fs, baseDir, targetPath, fs.mRootFolder, cachefile, exts);
		}
	}


	/**
	 * ファイルシステムへノードを追加
	 *
	 * @param file			キャッシュ対象ファイル
	 * @param folder		追加先フォルダ
	 * @param cachefile		ファイルをキャッシュするか
	 */
	private static void addFileSystem(FileSystem fs, String baseDir, String path, ObjectFolder folder, boolean cachefile, String[] exts) {
		File file = new File(baseDir, path);

		if (file.isDirectory()) {
			folder = FileSystem.getFolder(folder, file.getName());
			String[] files = file.list();
			for (int i = 0; i < files.length; i++) {
				ObjectFolder targetFolder = folder;
				addFileSystem(fs, baseDir, path + FileSystem.SEPARATOR + files[i], targetFolder, cachefile, exts);
			}
		} else {
			//拡張子による制限
			if (exts != null) {
				if (!isMatchExtention(file.getName(), exts))
					return;
			}
			byte[] filedata = null;
			if (file.exists() && cachefile) {
				try {
					if (log.isTraceEnabled())
						log.trace("add" + file.getAbsolutePath());

					filedata = createCache(file);
				} catch (Exception e) {
					filedata = null;
				}
			}
			fs.addFile(path, filedata);
		}
	}

	/**
	 * 拡張子が一致するか確認
	 * @param s
	 * @param exts
	 * @return
	 */
	private static boolean isMatchExtention(String s, String[] exts) {
		int index = s.lastIndexOf('.');
		if (index >= 0 && s.length() - 1 > index) {
			String ext = s.substring(s.lastIndexOf('.') + 1);
			//            boolean hit = false;
			for (int i = 0; i < exts.length; i++) {
				if (exts[i] != null && exts[i].equalsIgnoreCase(ext)) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * 入力ストリームからキャッシュを作成
	 * @param is
	 * @return
	 */
	private static byte[] createCache(File file) {
		FileInputStream is = null;
		try {
			is = new FileInputStream(file);
			if (file.length() > Integer.MAX_VALUE) {
				log.error("ファイルが大きすます" + file + " "+ file.length());
				return null;
			}
			ByteArrayOutputStream out = new ByteArrayOutputStream((int)file.length());
			byte[] buf = new byte[4096];
			while (true) {
				int len = is.read(buf);
				if (len == (-1))
					break;
				out.write(buf, 0, len);
			}
			out.flush();
			out.close();
			return out.toByteArray();
		} catch (IOException e) {
			log.error("ファイル読込エラー", e);
		} finally {
			try {
				is.close();
			} catch (IOException e) {
				log.error("ファイル読込エラー", e);
			}
		}
		return null;
	}
}
