package jp.kitec.lib.util.tree;

import java.util.NoSuchElementException;



/**
 * ツリーなどの階層構造におけるノード。<BR>
 *
 * @since 　2002/12/20
 * @author　Kawae
 * @version 2002/12/20
 *
 * Copyright (c) 2002 KITec Inc,.. All rights reserved.
 */
public class ObjectNode {

	/////////////////////////////////////////////////////////////////////////
	// フィールド

	/** ノード名 */
	private String			mName;
	/** ノードに対応するデータ */
	private Object			mObject;
	/** このノードの親 */
	private ObjectFolder	mParent;


	/**
	 * 指定された名前、親、およびデータを持つ新たなノードを構築する。
	 *
	 * @param name		このノードの名前
	 * @param parent	このノードの親
	 * @param o		このノードに対応するデータ
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public ObjectNode(String name, Object o) {
//	    this(name, o, null);
		mName   = name;
		mObject = o;
	}
//	public ObjectNode(String name, Object o, ObjectFolder parent) {
//		mName   = name;
//		mObject = o;
//		mParent = parent;
//	}

	/**
	 * このノードの名前を返す。
	 *
	 * @return ノードの名前
	 * @see #setName(String)
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public String getName() {
		return mName;
	}

	/**
	 * 指定された名前を、このノードの名前として設定する。
	 *
	 * @param ノードの名前
	 * @see #getName()
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public void setName(String name) {
		mName = name;
	}

	/**
	 * このノードに対応するデータを返す。
	 *
	 * @return このノードに対応するデータ
	 * @see #setObject(Object)
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public Object getObject() {
		return mObject;
	}

	/**
	 * 指定されたデータを、このノードに対応するデータとして設定する。
	 *
	 * @param このノードに対応するデータ
	 * @see #getObject()
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public void setObject(Object o) {
		mObject = o;
	}

	/**
	 * このノードの親を返す。
	 *
	 * @return ノードの親
	 * @see #setParent(ObjectFolder)
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public ObjectFolder getParent() {
		return mParent;
	}

	/**
	 * 指定された ObjectFolder を、このノードの親として設定する。
	 *
	 * @param ノードの親とする ObjectFolder
	 * @see #getParent()
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public void setParent(ObjectFolder f) {
		mParent = f;
	}


	/**
	 * Stringの取得
	 * 存在しない場合はnullを返す
	 */
	public String getString() {
		return (String)getObject();
	}


	/**
	 * Stringの取得
	 * 存在しない場合はnullを返す
	 *
	 * @param defalutValue
	 */
	public String getString(String defalutValue) {
		if (getObject() != null && getObject() instanceof String) {
			return (String)getObject();
		}
		return defalutValue;
	}

	/**
	 * int値の取得
	 */
	public int getInt() {
		String res = getString();
		if (res == null)
			throw new NoSuchElementException();
		return Integer.parseInt(res);
	}

	/**
	 * int値の取得
	 * 存在しない場合はdefaultValueを返す
	 *
	 * @param defaultValue
	 */
	public int getInt(int defaultValue) {
		try {
			return getInt();
		} catch (RuntimeException e) {
			return defaultValue;
		}
	}

	/**
	 * float値の取得
	 *
	 */
	public float getFloat() {
		String res = getString();
		if (res == null)
			throw new NoSuchElementException();
		return Float.parseFloat(res);
	}

	/**
	 * float値の取得
	 * 存在しない場合はdefaultValueを返す
	 *
	 * @param defaultValue
	 */
	public float getFloat(float defaultValue) {
		try {
			return getFloat();
		} catch (RuntimeException e) {
			return defaultValue;
		}
	}

	/**
	 * double値の取得
	 *
	 */
	public double getDouble() {
		String res = getString();
		return Double.parseDouble(res);
	}

	/**
	 * double値の取得
	 * 存在しない場合はdefaultValueを返す
	 *
	 */
	public double getDouble(double defaultValue) {
		try {
			return getDouble();
		} catch (RuntimeException e) {
			return defaultValue;
		}
	}

	/**
	 * boolの取得
	 * 存在しない場合は0を返す
	 *
	 */
	public boolean getBool() {
		String res = getString().trim();
		if (res.equalsIgnoreCase("1"))
			return true;
		if (res.equalsIgnoreCase("0"))
			return false;
		throw new IllegalStateException();
	}

	/**
	 * boolの取得
	 * 存在しない場合はdefaultValueを返す
	 *
	 * @param defaultValue
	 */
	public boolean getBool(boolean defaultValue) {
		try {
			return getBool();
		} catch (RuntimeException e) {
			return defaultValue;
		}
	}

	/**
	 * ノードオブジェクトがStringか確認
	 * @return
	 */
	public boolean isStringObject() {
		return (getObject() != null && getObject() instanceof String);
	}

	/**
	 * ノードオブジェクトがintか確認
	 * @return
	 */
	public boolean isIntObject() {
		if (!isStringObject())
			return false;
		String str = getString();
		try {
			Integer.valueOf(str).intValue();
			return true;
		} catch (NumberFormatException e) {
			return false;
		}
	}
	/**
	 * オブジェクトがBooleanか確認
	 * 文字列かつ"1"or"0"か確認
	 * @return
	 */
	public boolean isBoolObject() {
		if (!isStringObject())
			return false;
		String str = getString();
		if (str != null) {
			str = str.trim();
			return (str.equalsIgnoreCase("1") || str.equalsIgnoreCase("0"));
		}
		return false;
	}
	/**
	 * ノードオブジェクトがfloatか確認
	 * @return
	 */
	public boolean isFloatObject() {
		if (!isStringObject())
			return false;
		String str = getString();
		try {
			Float.parseFloat(str);
			return true;
		} catch (NumberFormatException e) {
			return false;
		}
	}
	/**
	 * ノードオブジェクトがdoubleか確認
	 * @return
	 */
	public boolean isDoubleObject() {
		if (!isStringObject())
			return false;
		String str = getString();
		try {
			Double.parseDouble(str);
			return true;
		} catch (NumberFormatException e) {
			return false;
		}
	}

	/**
	 * Stringの取得
	 * 存在しない場合はnullを返す
	 *
	 * @param on
	 */
	public static String getNodeString(ObjectNode on) {
		return (String)on.getObject();
	}

	/**
	 * int値の取得
	 *
	 * @param on
	 */
	public static int getNodeInt(ObjectNode on) {
		String res = getNodeString(on);
		if (res == null)
			throw new NoSuchElementException();
		return Integer.parseInt(res);
	}

	/**
	 * float値の取得
	 *
	 * @param on
	 */
	public static float getNodeFloat(ObjectNode on) {
		String res = getNodeString(on);
		if (res == null)
			throw new NoSuchElementException();
		return Float.parseFloat(res);
	}

	/**
	 * double値の取得
	 *
	 * @param on
	 */
	public static double getNodeDouble(ObjectNode on) {
		String res = getNodeString(on);
		return Double.parseDouble(res);
	}

	/**
	 * boolの取得
	 * 存在しない場合は0を返す
	 *
	 * @param on
	 */
	public static boolean getNodeBool(ObjectNode on) {
		String res = getNodeString(on).trim();
		if (res.equalsIgnoreCase("1"))
			return true;
		if (res.equalsIgnoreCase("0"))
			return false;
		throw new IllegalStateException();
	}

	/**
	 * ノードオブジェクトがStringか確認
	 * @param on
	 * @return
	 * @author kawae
	 * @since 2004/11/05
	 */
	public static boolean isStringObject(ObjectNode on) {
		return (on != null && on.getObject() != null && on.getObject() instanceof String);
	}

	/**
	 * ノードオブジェクトがintか確認
	 * @param on
	 * @return
	 * @author kawae
	 * @since 2004/11/05
	 */
	public static boolean isIntObject(ObjectNode on) {
		if (!isStringObject(on))
			return false;
		String str = getNodeString(on);
		try {
			Integer.valueOf(str).intValue();
			return true;
		} catch (NumberFormatException e) {
			return false;
		}
	}
	/**
	 * オブジェクトがBooleanか確認
	 * 文字列かつ"1"or"0"か確認
	 * @param on
	 * @return
	 * @author kawae
	 * @since 2004/11/05
	 */
	public static boolean isBoolObject(ObjectNode on) {
		if (!isStringObject(on))
			return false;
		String str = getNodeString(on);
		if (str != null) {
			str = str.trim();
			return (str.equalsIgnoreCase("1") || str.equalsIgnoreCase("0"));
		}
		return false;
	}
	/**
	 * ノードオブジェクトがfloatか確認
	 * @param on
	 * @return
	 * @author kawae
	 * @since 2004/11/05
	 */
	public static boolean isFloatObject(ObjectNode on) {
		if (!isStringObject(on))
			return false;
		String str = getNodeString(on);
		try {
			Float.parseFloat(str);
			return true;
		} catch (NumberFormatException e) {
			return false;
		}
	}
	/**
	 * ノードオブジェクトがdoubleか確認
	 * @param on
	 * @return
	 */
	public static boolean isDoubleObject(ObjectNode on) {
		if (!isStringObject(on))
			return false;
		String str = getNodeString(on);
		try {
			Double.parseDouble(str);
			return true;
		} catch (NumberFormatException e) {
			return false;
		}
	}
}
