/*
 * Copyright (c) 2007 KITec Inc,.. All rights reserved.
 */
package jp.kitec.lib.util;

import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.Locale;
import java.util.TimeZone;



/**
 * 日付ユーティリティクラス
 *
 * @author $Author$
 * @version $Revision$ $Date::                           $
 */
public class DateUtil {

	//------------------------------------------------------------------
	//- constants
	//------------------------------------------------------------------

	/** １日あたりの秒数 */
	public static final int SECOND_OF_DAY = 86400;

	/** YYYYMM形式のフォーマット */
	public static final String FORMAT_YYYYMM = "yyyyMM";

	/** YYYYMMDD形式のフォーマット */
	public static final String FORMAT_YYYYMMDD = "yyyyMMdd";

	/** YYYYMMDDhhmmss形式のフォーマット */
	public static final String FORMAT_YYYYMMDDHHMMSS = "yyyyMMddHHmmss";

	/** YYYY/MM形式のフォーマット */
	public static final String FORMAT_YYYY_MM = "yyyy/MM";

	/** YYYY/MM/DD形式のフォーマット */
	public static final String FORMAT_YYYY_MM_DD = "yyyy/MM/dd";

	/** YYYY/MM/DD hh:mm:ss形式のフォーマット */
	public static final String FORMAT_YYYY_MM_DD_HH_MM_SS = "yyyy/MM/dd HH:mm:ss";

	/** log4j形式（YYYY-MM-DD hh:mm:ss,SSS）のフォーマット */
	public static final String FORMAT_LOG4J = "yyyy-MM-dd HH:mm:ss,SSS";



	//------------------------------------------------------------------
	//- constructors
	//------------------------------------------------------------------

	/**
	 * コンストラクタ（外部new禁止）
	 */
	protected DateUtil() {
	}



	//-----------------------------------------------------------------------
	//- methods
	//-----------------------------------------------------------------------

	/**
	 * Timestampを指定したフォーマットで変換し日付文字列を生成します。
	 * フォーマットの詳細はSimpleDateFormatの説明を参照してください。
	 *
	 * @param format 変換用フォーマット
	 * @param timestamp 変換元Timestampオブジェクト
	 * @return 変換に成功した場合は日付文字列。できなかった場合はnullを返す。
	 */
	public static String form(String format, Timestamp timestamp) {
		String result = null;
		try {
			SimpleDateFormat sdf = new SimpleDateFormat(format);
			result = sdf.format(timestamp);
		} catch (Exception e) {
			result = null;
		}
		return result;
	}

	/**
	 * YYYYMMDD形式の日付を生成します。
	 *
	 * @param timestamp Timestampオブジェクト
	 * @return 日付（YYYYMMDD）
	 */
	public static String formYMD(Timestamp timestamp) {
		return form(FORMAT_YYYYMMDD, timestamp);
	}

	/**
	 * YYYYMMDDHHMMSS形式の日付を生成します。
	 *
	 * @param timestamp Timestampオブジェクト
	 * @return 日付（YYYYMMDDHHMMSS）
	 */
	public static String formYMDHMS(Timestamp timestamp) {
		return form(FORMAT_YYYYMMDDHHMMSS, timestamp);
	}

	/**
	 * UTC形式の日付文字列を生成する
	 *
	 * @param timestamp Timestampインスタンス
	 * @return UTC形式の日付文字列
	 */
	public static String formUTC(Timestamp timestamp) {
		SimpleDateFormat sdf = new SimpleDateFormat("dd-MMM-yyyy (EEE) HH:mm", Locale.US);
		sdf.setTimeZone(TimeZone.getTimeZone("Etc/UTC"));
		return sdf.format(timestamp) + " UTC";
	}

	/**
	 * GMT形式の日付文字列を生成する
	 *
	 * @param timestamp Timestampインスタンス
	 * @return GMT形式の日付文字列
	 */
	public static String formGMT(Timestamp timestamp) {
		SimpleDateFormat sdf = new SimpleDateFormat("EEE, dd-MMM-yyyy HH:mm:ss", Locale.US);
		sdf.setTimeZone(TimeZone.getTimeZone("Etc/GMT"));
		return sdf.format(timestamp) + " GMT";
	}



} // end-class
