/*
 * Copyright (c) 2009-2010 KITec Inc,.. All rights reserved.
 */
package jp.kitec.lib.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import jp.kitec.lib.util.tree.ObjectFolder;
import jp.kitec.lib.util.tree.ObjectNode;




/**
 * ObjectFolderユーティリティクラス
 *
 * @author $Author$
 * @version $Revision$ $Date::                           $
 */
public class ObjectFolderUtil {

	//------------------------------------------------------------------
	//- constructors
	//------------------------------------------------------------------

	/**
	 * コンストラクタ
	 */
	protected ObjectFolderUtil() {
	}



	//------------------------------------------------------------------
	//- methods
	//------------------------------------------------------------------

	/**
	 * ObjectFolder配下のObjectFolderを集める。
	 *
	 * @param folder ObjectFolder
	 * @return ObjectFolderコレクション
	 */
	public static Collection<ObjectFolder> collectChildFolders(ObjectFolder folder) {
		return collectChildFolders(folder, -1);
	}

	/**
	 * ObjectFolder配下のObjectFolderを集める。
	 *
	 * @param folder ObjectFolder
	 * @param depth 探索する深さ。-1の場合全探索を行う。
	 * @return ObjectFolderコレクション
	 */
	public static Collection<ObjectFolder> collectChildFolders(ObjectFolder folder, int depth) {
		List<ObjectFolder> collectFolderList = new ArrayList<ObjectFolder>();
		if (depth == 0) return collectFolderList;

		List<ObjectNode> childNodeList = folder.getChildren();
		for (ObjectNode childNode: childNodeList) {
			if (!(childNode instanceof ObjectFolder)) continue;
			ObjectFolder childFolder = (ObjectFolder)childNode;

			collectFolderList.add(childFolder);
			collectFolderList.addAll(collectChildFolders(childFolder, depth - 1));
		}

		return collectFolderList;
	}

	/**
	 * ObjectFolder配下から、名称が一致するObjectFolderを探す。
	 *
	 * @param folder ObjectFolder
	 * @param name ObjectFolder名
	 * @return ObjectFolderコレクション
	 */
	public static Collection<ObjectFolder> findChildFolders(ObjectFolder folder, String name) {
		return findChildFolders(folder, -1, name);
	}

	/**
	 * ObjectFolder配下から、名称が一致するObjectFolderを探す。
	 *
	 * @param folder ObjectFolder
	 * @param depth 探索する深さ。-1の場合全探索を行う。
	 * @param name ObjectFolder名
	 * @return ObjectFolderコレクション
	 */
	public static Collection<ObjectFolder> findChildFolders(ObjectFolder folder, int depth, String name) {
		Collection<ObjectFolder> childFolders = collectChildFolders(folder, depth);

		List<ObjectFolder> collectFolderList = new ArrayList<ObjectFolder>();
		for (ObjectFolder childFolder: childFolders) {
			if (!childFolder.getName().equals(name)) continue;
			collectFolderList.add(childFolder);
		}

		return collectFolderList;
	}

	/**
	 * ObjectFolder配下から、名称が一致するObjectNodeを探す。<br/>
	 * パスは"/"区切りで階層を表し、@で始まるパスをObjectNodeと見なす。<br/>
	 * ex. /path/to/@attr
	 *
	 * @param folder ObjectFolder
	 * @param path パス
	 * @return ObjectNode
	 */
	public static ObjectNode findChildNode(ObjectFolder parentFolder, String path) {
		String[] splitPaths = path.split("/");
		ObjectFolder currentFolder = parentFolder;

		for (String splitPath: splitPaths) {
			if (StringUtil.isEmpty(splitPath)) continue;
			if (splitPath.matches("^@.*")) {
				return currentFolder.findNode(splitPath.substring(1));
			}

			Collection<ObjectFolder> currentChildFolders = collectChildFolders(currentFolder, 1);
			for (ObjectFolder currentChildFolder: currentChildFolders) {
				if (currentChildFolder.getName().equals(splitPath)) {
					currentFolder = currentChildFolder;
					break;
				}
			}
		}

		return null;
	}



} // end-class
