package jp.kitec.lib.util;

import java.util.ArrayList;


/**
 * 履歴を管理する。<BR>
 *
 * @since 　2002/12/20
 * @author　Kawae
 * @version 2002/12/20
 *
 * Copyright (c) 2002 KITec Inc,.. All rights reserved.
 */
public class History<E, H> {

	/////////////////////////////////////////////////////////////////////////
	// フィールド

	/** 保存する履歴の最大数 */
	private int _maxHistoryNum = 8;
	private ArrayList<H> _history;
	private H _current;
	private HistoryDataConverter<E, H> _converter;


	/**
	 * 指定された数の履歴を保存する新たな履歴を構築する。
	 *
	 * @param hisnum	履歴の最大保存数
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public History(int hisnum, HistoryDataConverter<E, H> converter) {
		setMaxHistory(hisnum);
		_history = new ArrayList<H>();
		_current = null;
		_converter = converter;
	}

	/**
	 * 指定された数を、履歴の最大保存数として設定する。
	 *
	 * @param num	履歴の最大保存数
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 * @throws IllegalArgumentException
	 */
	public void setMaxHistory(int num) {
		if (num < 0)
			throw new IllegalArgumentException();
		_maxHistoryNum = num;
	}

	/**
	 * 現在、設定されている履歴の最大保存数を返す。
	 *
	 * @return	履歴の最大保存数
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public int getMaxHistory() {
		return _maxHistoryNum;
	}

	/**
	 * 現在、保存されているすべての履歴を削除する。
	 *
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public void flushHistory() {
		_current = null;
		_history.clear();
	}

	/**
	 * 現在、保存されている履歴を数を返す。
	 *
	 * @return 履歴の数
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public int getHistorySize() {
		return _history.size();
	}

	/**
	 * 新たに履歴を追加する。
	 * 保存している履歴の数が、履歴の最大保存数を超えた場合、
	 * 古い履歴は、順次削除される。
	 *
	 * @param o 追加する履歴
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public void addHistory(E o) {
		H b = _converter.writeHistory(o);
		if (b == null)
			return;
		if (_current != null) {
			int num = _history.indexOf(_current);
			if (num >= 0 && num < _history.size() - 1) {
				while(num + 1 <= _history.size() - 1)
					_history.remove(num + 1);
			}
		}
		_current = b;
		_history.add(b);
		while(_history.size() > _maxHistoryNum)
			_history.remove(0);
	}

	/**
	 * 保存されている履歴からアンドゥを行えるか否か判定する。
	 * 保存されている履歴が１つ以上存在する、かつ前回のアンドゥ、
	 * またはリドゥなどで処理された履歴が、最古の履歴でない場合のみ
	 * アンドゥ可能となる。
	 *
	 * @return アンドゥできる場合、true。できない場合は、false
	 * @since   2010/04/08
	 * @author  Kawae
	 */
	public boolean hasUndo() {
		if (_current == null)
			return false;
		if (_history.size() == 0)
			return false;

		int num = _history.indexOf(_current);

		if (num > 0)
			return true;
		return false;
	}

	/**
	 * 保存されている履歴からリドゥを行えるか否か判定する。
	 * 保存されている履歴が１つ以上存在する、かつ前回のアンドゥ、
	 * またはリドゥなどで処理された履歴が、最新の履歴でない場合のみ
	 * リドゥ可能となる。
	 *
	 * @return リドゥできる場合、true。できない場合は、false
	 * @since   2010/04/08
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public boolean hasRedo() {
		if (_current == null)
			return false;
		int num = _history.indexOf(_current);
		if (num >= 0 && num < _history.size() - 1)
			return true;
		return false;
	}

	/**
	 * 現在、保存されている履歴から最新の履歴を返す。
	 *
	 * @return 最新の履歴。履歴が存在しない場合は、null
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public H getLastHistory() {
		if (_history.size() == 0)
			return null;
		return _history.get(_history.size() - 1);
	}

	/**
	 * 現在の履歴を返す。
	 *
	 * @return 現在の履歴。
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public H getCurrentHistory() {
		return _current;
	}

	/**
	 * 現在の履歴の１世代前の履歴を返す。
	 *
	 * @return １世代前の履歴。
	 * @see #redo()
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public E undo() {
		if (_current == null)
			return null;
		int num = _history.indexOf(_current);
		if (num < 0)
			return null;
		if (num > 0) {
			_current = _history.get(num - 1);
			return _converter.readHistory(_current);
		}
		return null;
	}

	/**
	 * 現在の履歴の１世代後の履歴を返す。
	 *
	 * @return １世代後の履歴。
	 * @see #undo()
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public E redo() {
		if (_current == null)
			return null;
		int num = _history.indexOf(_current);
		if (num < 0)
			return null;
		if (num >= 0 && num < _history.size() - 1) {
			_current = _history.get(num + 1);
			return _converter.readHistory(_current);
		}
		return null;
	}
	
	public HistoryDataConverter<E, H> getConverter() {
		return _converter;
	}
}
