/*
 * Copyright (c) 2007-2009 KITec Inc,.. All rights reserved.
 */
package jp.kitec.lib.util;

import java.io.UnsupportedEncodingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;



/**
 * 文字チェッククラス
 *
 * @author $Author: tanaka $
 * @version $Revision: 18 $ $Date:: 2009-12-10 14:19:42 +0900#$
 */
public class CharChecker {

	//------------------------------------------------------------------
	//- constants
	//------------------------------------------------------------------

	/** ASCII・制御コード */
	public static final int[][] ASCII_CONTROL = {
		{0x00, 0x1f},
		{0x7f, 0x7f}
	};

	/** ASCII・数字 */
	public static final int[][] ASCII_NUMBER = {
		{0x30, 0x39}
	};

	/** ASCII・小数 */
	public static final int[][] ASCII_DECIMAL = {
		{0x30, 0x39},
		{0x2e, 0x2e}
	};

	/** ASCII・大英字 */
	public static final int[][] ASCII_CAPITAL_ALPHABET = {
		{0x41, 0x5a}
	};

	/** ASCII・小英字 */
	public static final int[][] ASCII_SMALL_ALPHABET = {
		{0x61, 0x7a}
	};

	/** ASCII・記号 */
	public static final int[][] ASCII_SIGN = {
		{0x20, 0x2f},
		{0x3a, 0x40},
		{0x5b, 0x60},
		{0x7b, 0x7e}
	};

	/** ASCII・英字 */
	public static final int[][][] ASCII_ALPHABET = {
		ASCII_CAPITAL_ALPHABET,
		ASCII_SMALL_ALPHABET
	};

	/** ASCII */
	public static final int[][][] ASCII = {
		ASCII_NUMBER,
		ASCII_CAPITAL_ALPHABET,
		ASCII_SMALL_ALPHABET,
		ASCII_SIGN
	};

	/** SJIS・半角カタカナ */
	public static final int[][] SJIS_HAN_KATAKANA = {
		{0xa1, 0xdf}
	};

	/** SJIS・半角文字 */
	public static final int[][][] SJIS_1BYTE = {
		ASCII_CONTROL,
		ASCII_NUMBER,
		ASCII_CAPITAL_ALPHABET,
		ASCII_SMALL_ALPHABET,
		ASCII_SIGN,
		SJIS_HAN_KATAKANA
	};

	/** SJIS・全角文字上位1byte */
	public static final int[][] SJIS_ZEN_UPPER = {
		{0x81, 0x9f},
		{0xe0, 0xef}
	};

	/** SJIS・全角文字下位1byte */
	public static final int[][] SJIS_ZEN_LOWER = {
		{0x40, 0x7e},
		{0x80, 0xfc}
	};

	/** SJIS・全角数字 */
	public static final int[][] SJIS_ZEN_NUMBER = {
		{0x824f, 0x8258}
	};

	/** SJIS・全角小数 */
	public static final int[][] SJIS_ZEN_DECIMAL = {
		{0x824f, 0x8258},
		{0x8144, 0x8144}
	};

	/** SJIS・全角大英字 */
	public static final int[][] SJIS_ZEN_CAPITAL_ALPHABET = {
		{0x8260, 0x8279}
	};

	/** SJIS・全角小英字 */
	public static final int[][] SJIS_ZEN_SMALL_ALPHABET = {
		{0x8281, 0x829a}
	};

	/** SJIS・全角ひらがな */
	public static final int[][] SJIS_ZEN_HIRAGANA = {
		{0x829f, 0x82f1}
	};

	/** SJIS・全角カタカナ */
	public static final int[][] SJIS_ZEN_KATAKANA = {
		{0x8340, 0x8396}
	};

	/** SJIS・全角棒引き（長音符）*/
	public static final int[][] SJIS_ZEN_BOU = {
		{0x815b, 0x815b}
	};

	/** SJIS・全角記号 */
	public static final int[][] SJIS_ZEN_SIGN = {
		{0x8140, 0x81ac}, //記号
		{0x81b8, 0x81bf}, //記号
		{0x81c8, 0x81ce}, //記号
		{0x81da, 0x81e8}, //記号
		{0x81f0, 0x81f7}, //記号
		{0x81fc, 0x81fc}, //丸
		{0x839f, 0x83b6}, //ギリシャ大文字
		{0x83bf, 0x83d6}, //ギリシャ小文字
		{0x8440, 0x8460}, //ロシア大文字
		{0x8470, 0x8491}, //ロシア小文字
		{0x849f, 0x84be}  //罫線
	};

	/** SJIS・全角第一水準漢字 */
	public static final int[][] SJIS_ZEN_KANJI1 = {
		{0x889f, 0x9872}
	};

	/** SJIS・全角第二水準漢字 */
	public static final int[][] SJIS_ZEN_KANJI2 = {
		{0x989f, 0x9ffc},
		{0xe040, 0xeaa4}
	};

	/** SJIS・全角漢字 */
	public static final int[][][] SJIS_ZEN_KANJI = {
		SJIS_ZEN_KANJI1,
		SJIS_ZEN_KANJI2
	};

	/** SJIS・全角文字 */
	public static final int[][][] SJIS_2BYTE = {
		SJIS_ZEN_NUMBER,
		SJIS_ZEN_CAPITAL_ALPHABET,
		SJIS_ZEN_SMALL_ALPHABET,
		SJIS_ZEN_HIRAGANA,
		SJIS_ZEN_KATAKANA,
		SJIS_ZEN_SIGN,
		SJIS_ZEN_KANJI1,
		SJIS_ZEN_KANJI2
	};

	/** SJIS・全文字 */
	public static final int[][][] SJIS_ALL = {
		ASCII_CONTROL,
		ASCII_NUMBER,
		ASCII_CAPITAL_ALPHABET,
		ASCII_SMALL_ALPHABET,
		ASCII_SIGN,
		SJIS_HAN_KATAKANA,
		SJIS_ZEN_NUMBER,
		SJIS_ZEN_CAPITAL_ALPHABET,
		SJIS_ZEN_SMALL_ALPHABET,
		SJIS_ZEN_HIRAGANA,
		SJIS_ZEN_KATAKANA,
		SJIS_ZEN_SIGN,
		SJIS_ZEN_KANJI1,
		SJIS_ZEN_KANJI2
	};

	/** MailAddress使用可能文字文字列 */
	public static final int[][][] MAIL_ADDRESS_SET = {
		ASCII_NUMBER,
		ASCII_CAPITAL_ALPHABET,
		ASCII_SMALL_ALPHABET,
		new int[][]{
			{'!', '!'},
			{'#', '#'},
			{'$', '$'},
			{'%', '%'},
			{'&', '&'},
			{'\'', '\''},
			{'*', '*'},
			{'+', '+'},
			{'-', '-'},
			{'/', '/'},
			{'=', '='},
			{'?', '?'},
			{'^', '^'},
			{'_', '_'},
			{'`', '`'},
			{'{', '{'},
			{'|', '|'},
			{'}', '}'},
			{'~', '~'}
		}
	};

	/** Windowsファイル名使用不可能文字文字列 */
	public static final int[][] WIN_FILE_NAME_EXCLUDE = {
		{'\'', '\''},
		{'!', '!'},
		{'"', '"'},
		{'|', '|'},
		{'*', '*'},
		{'<', '<'},
		{'>', '>'},
		{'?', '?'},
		{'\\', '\\'},
		{';', ';'},
		{':', ':'},
		{',', ','},
		{'/', '/'}
	};



	//------------------------------------------------------------------
	//- constants
	//------------------------------------------------------------------

	/** ログ */
	private static final Log log = LogFactory.getLog(CharChecker.class);



	//------------------------------------------------------------------
	//- constructors
	//------------------------------------------------------------------

	/**
	 * コンストラクタ（外部new禁止）
	 */
	protected CharChecker() {
	}



	//------------------------------------------------------------------
	//- methods
	//------------------------------------------------------------------

	/**
	 * target 文字列が pattern で構成されているかチェックする
	 *
	 * @param target チェック対象文字列
	 * @param patternSet 許可する文字範囲の配列
	 * @return pattern文字列で構成されていればtrue。
	 */
	public static boolean hasCharacter(String target, int[][][] patternSet) {
		boolean result = true;
		if (target == null || target.compareTo("") == 0) return true;

		for (int i = 0; i < target.length(); i++) {
			if (!hasCharacter(target.charAt(i), patternSet)) {
				result = false;
				break;
			}
		}

		return result;
	}

	/**
	 * target 文字が pattern で構成されているかチェックする
	 *
	 * @param target チェック対象文字
	 * @param patternSet 許可する文字範囲の配列
	 * @return pattern文字列で構成されていればtrue。
	 */
	public static boolean hasCharacter(char target, int[][][] patternSet) {
		for (int i = 0; i < patternSet.length; i++) {
			int[][] patterns = patternSet[i];
			if (hasCharacter(target, patterns)) return true;
		}
		return false;
	}

	/**
	 * target 文字列が pattern で構成されているかチェックする
	 *
	 * @param target チェック対象文字列
	 * @param patterns 許可する文字範囲の配列
	 * @return pattern文字列で構成されていればtrue。
	 */
	public static boolean hasCharacter(String target, int[][] patterns) {
		for (int i = 0; i < target.length(); i++) {
			if (!hasCharacter(target.charAt(i), patterns)) return false;
		}
		return true;
	}

	/**
	 * target 文字が pattern で構成されているかチェックする
	 *
	 * @param target チェック対象文字
	 * @param patterns 許可する文字範囲の配列
	 * @return pattern文字列で構成されていればtrue。
	 */
	public static boolean hasCharacter(char target, int[][] patterns) {
		String targetStr = Character.toString(target);
		byte[] targetBytes = null;
		try {
			targetBytes = targetStr.getBytes("SJIS");
		} catch (UnsupportedEncodingException e) {
			log.info("", e);
			return false;
		}

		int code = 0;
		int b1 = targetBytes[0] & 0x000000ff;
		if (isSJIS_1byte(b1)) {
			code = b1;
		} else {
			int b2 = targetBytes[1] & 0x000000ff;
			if (!isSJIS_2byte(b1, b2)) return false;
			code = ((b1 << 8) & 0x0000ff00) | b2;
		}

		boolean match = false;
		for (int j = 0; j < patterns.length; j++) {
			int[] pattern = patterns[j];
			if (pattern[0] <= code &&  code <= pattern[1]) match = true;
		}
		return match;
	}

	/**
	 * target 文字列が pattern 以外で構成されているかチェックする
	 *
	 * @param target チェック対象文字
	 * @param pattern 許可しない文字範囲
	 * @return pattern文字列以外で構成されていればtrue。
	 */
	public static boolean hasNotCharacter(String target, int[][][] patternSet) {
		boolean result = true;
		if (target == null || target.compareTo("") == 0) return true;

		for (int i = 0; i < target.length(); i++) {
			if (!hasNotCharacter(target.charAt(i), patternSet)) {
				result = false;
				break;
			}
		}

		return result;
	}

	/**
	 * target 文字が pattern 以外で構成されているかチェックする
	 *
	 * @param target チェック対象文字
	 * @param pattern 許可しない文字リスト
	 * @return pattern文字列以外で構成されていればtrue。
	 */
	public static boolean hasNotCharacter(char target, int[][][] patternSet) {
		for (int i = 0; i < patternSet.length; i++) {
			int[][] patterns = patternSet[i];
			if (!hasNotCharacter(target, patterns)) return false;
		}
		return true;
	}

	/**
	 * target 文字列が pattern 以外で構成されているかチェックする
	 *
	 * @param target チェック対象文字
	 * @param pattern 許可しない文字範囲
	 * @return pattern文字列以外で構成されていればtrue。
	 */
	public static boolean hasNotCharacter(String target, int[][] patterns) {
		for (int i = 0; i < target.length(); i++) {
			if (hasNotCharacter(target.charAt(i), patterns)) return true;
		}
		return false;
	}

	/**
	 * target 文字が pattern 以外で構成されているかチェックする
	 *
	 * @param target チェック対象文字
	 * @param pattern 許可しない文字リスト
	 * @return pattern文字列以外で構成されていればtrue。
	 */
	public static boolean hasNotCharacter(char target, int[][] patterns) {
		String targetStr = Character.toString(target);
		byte[] targetBytes = null;
		try {
			targetBytes = targetStr.getBytes("SJIS");
		} catch (UnsupportedEncodingException e) {
			log.info("", e);
			return true;
		}

		int code = 0;
		int b1 = targetBytes[0] & 0x000000ff;
		if (isSJIS_1byte(b1)) {
			code = b1;
		} else {
			int b2 = targetBytes[1] & 0x000000ff;
			if (!isSJIS_2byte(b1, b2)) return true;
			code = ((b1 << 8) & 0x0000ff00) | b2;
		}

		for (int i = 0; i < patterns.length; i++) {
			int[] patternPair = patterns[i];
			if (patternPair[0] <= code && code <= patternPair[1]) return false;
		}
		return true;
	}

	/**
	 * 1byte ShiftJIS文字かチェックする。
	 *
	 * @param b1 1byte目
	 * @return 1byte ShiftJISの場合true。
	 */
	private static boolean isSJIS_1byte(int b1) {
		for (int i = 0; i < SJIS_1BYTE.length; i++) {
			int[][] patterns = SJIS_1BYTE[i];
			for (int j = 0; j < patterns.length; j++) {
				int[] pattern = patterns[j];
				if (pattern[0] <= b1 && b1 <= pattern[1]) return true;
			}
		}
		return false;
	}

	/**
	 * 2byte ShiftJIS文字かチェックする。
	 *
	 * @param b1 1byte目
	 * @param b2 2byte目
	 * @return 2byte ShiftJISの場合true。
	 */
	private static boolean isSJIS_2byte(int b1, int b2) {
		boolean result = false;
		for (int i = 0; i < SJIS_ZEN_UPPER.length; i++) {
			int[] pattern = SJIS_ZEN_UPPER[i];
			if (pattern[0] <= b1 && b1 <= pattern[1]) {
				result = true;
				break;
			}
		}
		if (!result) return false;

		for (int i = 0; i < SJIS_ZEN_LOWER.length; i++) {
			int[] pattern = SJIS_ZEN_LOWER[i];
			if (pattern[0] <= b2 && b2 <= pattern[1]) {
				result = true;
				break;
			}
		}
		return result;
	}

	/**
	 * 半角カタカナだけで構成されている文字列かチェックする
	 *
	 * @param target チェック対象文字列
	 * @return 半角カタカナだけで構成されていればtrue。でなければfalse
	 */
	public static boolean isHanKata(String target) {
		return hasCharacter(target, SJIS_HAN_KATAKANA);
	}

	/**
	 * 半角カタカナ文字かチェックする
	 *
	 * @param target チェック対象文字
	 * @return 半角カタカナならばtrue。
	 */
	public static boolean isHanKata(char target) {
		return hasCharacter(target, SJIS_HAN_KATAKANA);
	}

	/**
	 * 全角カタカナ + "ー" だけで構成されている文字列かチェックする
	 *
	 * @param target チェック対象文字列
	 * @return 全角カタカナだけで構成されていればtrue。でなければfalse
	 */
	public static boolean isZenKata(String target) {
		return hasCharacter(target, new int[][][]{SJIS_ZEN_KATAKANA, SJIS_ZEN_BOU});
	}

	/**
	 * 全角カタカナ or "ー" 文字かチェックする
	 *
	 * @param target チェック対象文字
	 * @return 全角カタカナ or "ー" ならばtrue。
	 */
	public static boolean isZenKata(char target) {
		return hasCharacter(target, new int[][][]{SJIS_ZEN_KATAKANA, SJIS_ZEN_BOU});
	}

	/**
	 * 半角整数だけで構成されている文字列かチェックする
	 *
	 * @param target チェック対象文字列
	 * @return 半角整数だけで構成されていればtrue。でなければfalse
	 */
	public static boolean isHanNum(String target) {
		return hasCharacter(target, ASCII_NUMBER);
	}

	/**
	 * 半角整数文字かチェックする
	 *
	 * @param target チェック対象文字
	 * @return 半角整数ならばtrue。でなければfalse
	 */
	public static boolean isHanNum(char target) {
		return hasCharacter(target, ASCII_NUMBER);
	}

	/**
	 * 全角整数だけで構成されている文字列かチェックする
	 *
	 * @param target チェック対象文字列
	 * @return 全角整数だけで構成されていればtrue。でなければfalse
	 */
	public static boolean isZenNum(String target) {
		return hasCharacter(target, SJIS_ZEN_NUMBER);
	}

	/**
	 * 全角整数文字かチェックする
	 *
	 * @param target チェック対象文字
	 * @return 全角整数ならばtrue。
	 */
	public static boolean isZenNum(char target) {
		return hasCharacter(target, SJIS_ZEN_NUMBER);
	}

	/**
	 * 半角で構成されている文字列かチェックする
	 *
	 * @param target チェック対象文字列
	 * @return 半角で構成されていればtrue。でなければfalse
	 */
	public static boolean isHankaku(String target) {
		return hasCharacter(target, SJIS_1BYTE);
	}

	/**
	 * 半角かチェックする
	 *
	 * @param target チェック対象文字
	 * @return 半角ならばtrue。
	 */
	public static boolean isHankaku(char target) {
		return hasCharacter(target, SJIS_1BYTE);
	}

	/**
	 * 全角で構成されている文字列かチェックする
	 *
	 * @param target チェック対象文字列
	 * @return 全角で構成されていればtrue。でなければfalse
	 */
	public static boolean isZenkaku(String target) {
		return !isHankaku(target);
	}

	/**
	 * 全角かチェックする
	 *
	 * @param target チェック対象文字
	 * @return 全角ならばtrue。
	 */
	public static boolean isZenkaku(char target) {
		return !isHankaku(target);
	}

	/**
	 * Asciiで構成されている文字列かチェックする
	 *
	 * @param target チェック対象文字列
	 * @return Asciiで構成されていればtrue。
	 */
	public static boolean isAscii(String target) {
		return hasCharacter(target, ASCII);
	}

	/**
	 * Ascii文字かチェックする
	 *
	 * @param target チェック対象文字
	 * @return Ascii文字ならばtrue。
	 */
	public static boolean isAscii(char target) {
		return hasCharacter(target, ASCII);
	}



	/**
	 * 半角カタカナ以外で構成されている文字列かチェックする
	 *
	 * @param target チェック対象文字列
	 * @return 半角カタカナ以外で構成されていればtrue。
	 */
	public static boolean isNotHanKata(String target) {
		return hasNotCharacter(target, SJIS_HAN_KATAKANA);
	}

	/**
	 * 半角カタカナ以外の文字かチェックする
	 *
	 * @param target チェック対象文字
	 * @return 半角カタカナ以外ならばtrue。
	 */
	public static boolean isNotHanKata(char target) {
		return hasNotCharacter(target, SJIS_HAN_KATAKANA);
	}

	/**
	 * 全角カタカナ以外で構成されている文字列かチェックする
	 *
	 * @param target チェック対象文字列
	 * @return 全角カタカナ以外で構成されていればtrue。
	 */
	public static boolean isNotZenKata(String target) {
		return hasNotCharacter(target, SJIS_ZEN_KATAKANA);
	}

	/**
	 * 全角カタカナ以外の文字かチェックする
	 *
	 * @param target チェック対象文字列
	 * @return 全角カタカナ以外ならばtrue。
	 */
	public static boolean isNotZenKata(char target) {
		return hasNotCharacter(target, SJIS_ZEN_KATAKANA);
	}

	/**
	 * 半角整数以外で構成されている文字列かチェックする
	 *
	 * @param target チェック対象文字列
	 * @return 半角整数以外で構成されていればtrue。
	 */
	public static boolean isNotHanNum(String target) {
		return hasNotCharacter(target, ASCII_NUMBER);
	}

	/**
	 * 半角整数以外の文字かチェックする
	 *
	 * @param target チェック対象文字
	 * @return 半角整数以外ならばtrue。
	 */
	public static boolean isNotHanNum(char target) {
		return hasNotCharacter(target, ASCII_NUMBER);
	}

	/**
	 * 全角整数以外で構成されている文字列かチェックする
	 *
	 * @param target チェック対象文字列
	 * @return 全角整数以外で構成されていればtrue。
	 */
	public static boolean isNotZenNum(String target) {
		return hasNotCharacter(target, SJIS_ZEN_NUMBER);
	}

	/**
	 * 全角整数以外の文字かチェックする
	 *
	 * @param target チェック対象文字
	 * @return 全角整数以外ならばtrue。
	 */
	public static boolean isNotZenNum(char target) {
		return hasNotCharacter(target, SJIS_ZEN_NUMBER);
	}

	/**
	 * 半角以外で構成されている文字列かチェックする
	 *
	 * @param target チェック対象文字列
	 * @return 半角以外で構成されていればtrue。
	 */
	public static boolean isNotHankaku(String target) {
		return hasNotCharacter(target, SJIS_1BYTE);
	}

	/**
	 * 半角以外の文字かチェックする
	 *
	 * @param target チェック対象文字
	 * @return 半角以外ならばtrue。
	 */
	public static boolean isNotHankaku(char target) {
		return hasNotCharacter(target, SJIS_1BYTE);
	}



	/**
	 * 半角小数だけで構成されている文字列かチェックする
	 *
	 * @param target チェック対象文字列
	 * @return 半角小数だけで構成されていればtrue。
	 */
	public static boolean isHanDec(String target) {
		return isHanDec(target, -1, -1);
	}

	/**
	 * 半角小数だけで構成されている文字列かチェックする
	 * また、整数部／小数部の桁数チェックも行う
	 *
	 * @param target チェック対象文字列
	 * @param numCount 整数部の桁数（-1だとチェックしない）
	 * @param decCount 小数部の桁数（-1だとチェックしない）
	 * @return 半角小数だけで構成されていればtrue。
	 */
	public static boolean isHanDec(String target, int numCount, int decCount) {
		// 適合する文字だけかチェック
		if (!hasCharacter(target, ASCII_DECIMAL)) return false;

		// 小数点の数をチェック（０個か１個ならばOK）
		int point = 0;
		int count;
		for (count = 0; ; count++) {
			point = target.indexOf(".", point);
			if (point == -1) break;
			point++;
		}
		if (count >= 2) return false;

		// 小数点が文字列の最後かチェック
		if (count == 1) {
			if (target.indexOf(".") + 1 == target.length()) return false;
		}

		// 整数部、小数部が指定された桁数かチェックする
		if (count == 0) {
			if (numCount != -1 && target.length() > numCount) return false;
		} else {
			point = target.indexOf(".");
			if (numCount != -1 && target.substring(0, point).length() > numCount) return false;
			if (decCount != -1 && target.substring(point + 1).length() > decCount) return false;
		}

		return true;
	}

	/**
	 * 全角小数だけで構成されている文字列かチェックする
	 *
	 * @param target チェック対象文字列
	 * @return 全角小数だけで構成されていればtrue。
	 */
	public static boolean isZenDec(String target) {
		return hasCharacter(target, SJIS_ZEN_DECIMAL);
	}

	/**
	 * 機種全角整数以外で構成されている文字列かチェックする
	 *
	 * @param target チェック対象文字列
	 * @return 全角整数以外で構成されていればtrue。
	 *
	 *
	 *
	 * 機種依存以外の文字かチェックする
	 *
	 * @param target チェック対象文字
	 * @return 機種依存以外ならばtrue。
	 */
	public static boolean isNotPlatformDependent(String target) {
		return !isUnsupportedCharacter(target) && !hasNotCharacter(target, SJIS_ALL);
	}

	/**
	 * 機種依存以外の文字かチェックする
	 *
	 * @param target チェック対象文字
	 * @return 機種依存以外ならばtrue。
	 */
	public static boolean isNotPlatformDependent(char target) {
		return !isUnsupportedCharacter(target) && !hasNotCharacter(target, SJIS_ALL);
	}

	/**
	 * target 文字列が、この環境でサポートされていない文字を含んでいるかチェックする。
	 *
	 * @param target チェック対象文字列
	 * @return サポートされていない文字が含まれる場合true。
	 */
	public static boolean isUnsupportedCharacter(String target) {
		for (int i = 0; i < target.length(); i++) {
			if (isUnsupportedCharacter(target.charAt(i))) return true;
		}
		return false;
	}

	/**
	 * target 文字が、この環境でサポートされていないことをチェックする。
	 *
	 * @param target チェック対象文字
	 * @return サポートされていない文字の場合true。
	 */
	public static boolean isUnsupportedCharacter(char target) {
		String targetStr = Character.toString(target);
		byte[] targetBytes = null;
		try {
			targetBytes = targetStr.getBytes("SJIS");
		} catch (UnsupportedEncodingException e) {
			log.info("", e);
			return true;
		}

		int code = 0;
		int b1 = targetBytes[0] & 0x000000ff;
		if (isSJIS_1byte(b1)) {
			code = b1;
		} else {
			int b2 = targetBytes[1] & 0x000000ff;
			if (!isSJIS_2byte(b1, b2)) return true;
			code = ((b1 << 8) & 0x0000ff00) | b2;
		}

		return target != '?' && code == 0x3f;
	}

	/**
	 * Windowsファイル名使用可能文字で構成されている文字列かチェックする
	 *
	 * @param target チェック対象文字列
	 * @return Windowsファイル名使用可能文字で構成されていればtrue。
	 */
	public static boolean isWinFileName(String target) {
		return hasNotCharacter(target, WIN_FILE_NAME_EXCLUDE);
	}



} // end-class
