package jp.kitec.lib.io;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FilterInputStream;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;


/**
 * すべてのファイルを抽象化したクラス。<BR>
 * 
 * @since 　2002/12/20
 * @author　Kawae
 * @version 2002/12/20
 * 
 * Copyright (c) 2002 KITec Inc,.. All rights reserved.
 */
public class AbstFile {

	private static final Log log = LogFactory.getLog(AbstFile.class);
	/////////////////////////////////////////////////////////////////////////
	// 定数
	
	public static final String CODE_MS932 = "MS932";
	public static final String CODE_UTF8 = "UTF-8";
	public static final String CODE_EUCJP = "EUC_JP";
	public static final String CODE_AUTO = "JISAutoDetect";
	
	/**
	 * ファイル・パスの区切り文字
	 * @see java.awt.File#separatorChar
	 */
	public static final char SEPARATOR = File.separatorChar;
	/** ファイルの改行文字 */
	public static final String LN = "\r\n";

	/////////////////////////////////////////////////////////////////////////
	// フィールド

	/** 仮想パス */
	private String mPath;

	/** ファイル名　*/
	private String mFileName;
	
	/** データ */
	protected byte[] mData = null; 

	/**
	 * 空の AbstFile を構築する。
	 * 
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public AbstFile() {
	}

	/**
	 * 指定されたファイルを書き込み用に開く。
	 * 
	 * @param filename データを書き込むファイル名
	 * @return 指定されたファイルの書き込み用のストリーム。
	 * 			ファイルのオープンに失敗した場合は、null
	 * @see #closeNewFileToBuffer()
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */	
	public OutputStream openOutputStream(String filename) {
		mPath = getParent(filename);
		mFileName = getFile(filename);
		return new AbstFileOutputStream(this);
	}

	/**
	 * この AbstFile が保持するデータを読み込む
	 * ストリームを開く。
	 * 
	 * @return 読み込み用のストリーム。
	 * 			ストリームのオープンに失敗した場合は、null
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public InputStream openInputStream() {
		return new AbstFileInputStream(this);
	}

	/**
	 * この AbstFile が保持するデータを読み込む
	 * BufferedReaderを開く。
	 * 
	 * @return 読み込み用のBufferedReader
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public BufferedReader openBufferedReader() {
		return new BufferedReader(new InputStreamReader(openInputStream()));
	}

	/**
	 * この AbstFile が保持するデータを読み込む
	 * BufferedReaderを開く。
	 * 
	 * @param code 文字コード
	 * @return 読み込み用のBufferedReader
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public BufferedReader openBufferedReader(String code) throws UnsupportedEncodingException {
		return new BufferedReader(new InputStreamReader(openInputStream(), code));
	}

	/**
	 * 指定されたストリームを使用して、ファイルからデータを読み込み、
	 * 内部データへ保存する。
	 * 
	 * @param is	読み込み用ストリーム
	 * @param f	ファイル 
	 * @return 読み込みに成功した場合、true。
	 * 			読み込みに失敗した場合は、false。
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public synchronized boolean read(InputStream is, String file) {
		
		return read(is, file, 2048);
	}


	/**
	 * 指定されたストリームを使用して、ファイルからデータを読み込み、
	 * 内部データへ保存する。
	 * 
	 * @param is	読み込み用ストリーム
	 * @param f	ファイル 
	 * @param size	初期確保サイズ
	 * @return 読み込みに成功した場合、true。
	 * 			読み込みに失敗した場合は、false。
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public synchronized boolean read(InputStream is, String file, int size) {
		mPath = getParent(file);
		mFileName = getFile(file);

		try {
			/** バッファ */
			byte tmpbyte[] = new byte[2048];

			ByteArrayOutputStream out = new ByteArrayOutputStream(size);
			int count = 0;
			while(true) {
				try {
					count = 0;
					count = is.read(tmpbyte);
					if (count < 0)
						break;
					if (count > 0) {
						out.write(tmpbyte, 0, count);
						count = 0;
					}
				} catch (EOFException e) {
					out.write(tmpbyte, 0, count);
					break;
				} catch (IOException e) {
					out.write(tmpbyte, 0, count);
					log.error("I/Oエラー", e);
					break;
				}
			}
			out.close();
			mData = out.toByteArray();
			is.close();
		} catch (Exception e) {
			return false;
		}
		return true;
	}

	/**
	 * この AbstFile が保持する内部データを、指定されたストリームへ書き込む。
	 * 
	 * @param os	書き込む用ストリーム
	 * @throws	IOException	書き込みに失敗した場合
	 */
	public void write(OutputStream os) throws IOException {
		os.write(mData);
		os.flush();
	}
	
	/**
	 * この AbstFile が保持するデータを返す。
	 * 
	 * @return データ
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public byte[] getData() {
		return mData;
	}
	
	/**
	 * 現在、設定されているファイルのファイル名を返す。
	 * 
	 * @return ファイル名
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public String getName() {
		return mFileName;
	}

	/**
	 * この AbstFile の対象となっているファイル名を含む
	 * ファイルへのパスを返す。
	 * 
	 * @return パス
	 * @see #SEPARATOR
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public String getPath() {
		String path = "";
		if (mPath != null)
			path += (mPath + SEPARATOR);
		if (mFileName != null)
			path += mFileName;
		return path;
	}

	/**
	 * この AbstFile の対象となっているファイル名を除く
	 * ファイルまでのパスを返す。
	 * 
	 * @return パス
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public String getParent() {
		return mPath;
	}

	/**
	 * 新たなデータを設定する。
	 * それまでに保持していたデータは、すべて、
	 * 新たなデータに置き換えられる。
	 * 
	 * @param b	データ
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public void setData(byte[] b) {
		if (b == null)
			throw new NullPointerException();
		mData = b;
	}

	/**
	 * この AbstFile が保持するデータを書き込むファイルを設定する。
	 * 
	 * @param s ファイル名
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public void setFile(String s) {
		mPath = getParent(s);
		mFileName = getFile(s);
	}

	/**
	 * 指定されたパスの１階層上までのパスを返す。<BR>
	 * 例）<BR>
	 * 　　　引数　：c:\temp\aaa\bb\ff.txt<BR>
	 * 　　　戻り値：c:\temp\aaa\bb<BR>
	 * 
	 * @param s	パス
	 * @return １階層上までのパス
	 * 			指定されたパスにファイルの区切り文字が
	 * 			存在しない場合は、null。
	 * @see #SEPARATOR
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public static String getParent(String s) {
		s = changeSeparator(s);
		int idx = s.lastIndexOf(SEPARATOR);
		if (idx >= 0) 
			return s.substring(0, idx);
		return null;
	}

	/**
	 * 指定されたパスからファイル名を返す。<BR>
	 * 例）<BR>
	 * 　　　引数　：c:\temp\aaa\bb\ff.txt<BR>
	 * 　　　戻り値：ff.txt<BR>
	 * 
	 * @param s	パス
	 * @return ファイル名
	 * @see #SEPARATOR
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public static String getFile(String s) {
		s = changeSeparator(s);
		int idx = s.lastIndexOf(SEPARATOR);
		if (idx >= 0 && (idx + 1) < s.length() - 1) 
			return s.substring(idx + 1, s.length());
		return s;
	}

	/**
	 * 指定されたパスから拡張子を除いたファイル名を返す。<BR>
	 * 例）<BR>
	 * 　　　引数　：c:\temp\aaa\bb\ff.txt<BR>
	 * 　　　戻り値：ff<BR>
	 * 
	 * @param s	パス
	 * @return 拡張子を除いたファイル名
	 * @see #SEPARATOR
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public static String getFileNonExt(String s) {
		s = getFile(s);
		if (s == null)
			return null;
		int idx = s.lastIndexOf(".");
		if (idx >= 0 && idx < s.length() - 1) 
			return s.substring(0, idx);
		return null;
	}

	/**
	 * 指定されたパスとファイル名から新たなパスを作成する。<BR>
	 * 例）<BR>
	 * 　　引数 s1：c:\temp\aaa\bb<BR>
	 * 　　　　 s2：ff.txt<BR>
	 * 　　戻り値：c:\temp\aaa\bb\ff.txt<BR>
	 * なお、パスの区切り文字は、AbstFile のデフォルトの
	 * 区切り文字が使用される。
	 * 
	 * @param s1	パス
	 * @param s2	ファイル名
	 * @return 拡張子を除いたファイル名
	 * @see #SEPARATOR
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public static String getCreatePath(String s1, String s2) {
		s1 = changeSeparator(s1);
		while (true) {
			if (s1.lastIndexOf(SEPARATOR) == (s1.length() - 1))
				s1 = s1.substring(0, s1.length() - 1);
			else
				break;
		}
		s2 = changeSeparator(s2);
		while(true) {
			if (s2.charAt(0) == SEPARATOR)
				s2 = s2.substring(1);
			else
				break;
		}
		
		return s1 + SEPARATOR + s2;
	}
	
	/**
	 * 指定されたパスに存在するパスの区切り文字を変更する。
	 * AbstFile#SEPARATOR = '/' の場合、パスに存在する
	 * '\\' -> '/' となり、'\\'の場合は、'/' -> '\\'となる。
	 * 
	 * @param s	パス
	 * @see #SEPARATOR
	 * @return 区切りも文字が変換されたパス
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public static String changeSeparator(String s) {
		if (SEPARATOR != '/')
			s = s.replace('/', SEPARATOR);
		if (SEPARATOR != '\\')
			s = s.replace('\\', SEPARATOR);
		return s;
	}

	static class AbstFileInputStream extends FilterInputStream {
		/**
		 * @param in
		 */
		public AbstFileInputStream(AbstFile af) {
			super(new ByteArrayInputStream(af.getData()));
		}
	}

	static class AbstFileOutputStream extends FilterOutputStream {
		private AbstFile mFile;

		public AbstFileOutputStream(AbstFile af) {
			super(new ByteArrayOutputStream());
			mFile = af;
		}

		/**
		 * ストリームを閉じて、内容をAbstFileに反映する。
		 */
		@Override
		public void close() throws IOException {
			super.close();
			mFile.setData(((ByteArrayOutputStream)out).toByteArray());
		}
	}
}
