/*
 * Copyright (c) 2007-2009 KITec Inc,.. All rights reserved.
 */
package jp.kitec.lib.util;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.StringTokenizer;



/**
 * 文字列ユーティリティクラス
 *
 * @author $Author: tanaka $
 * @version $Revision: 22 $ $Date:: 2010-03-11 21:42:06 +0900#$
 */
public class StringUtil {

	//------------------------------------------------------------------
	//- constants
	//------------------------------------------------------------------

	/** WhiteSpace */
	public static char[] WHITE_SPACES = {0x0020, 0x3000, '\t'};



	//------------------------------------------------------------------
	//- constructors
	//------------------------------------------------------------------

	/**
	 * コンストラクタ（外部new禁止）
	 */
	protected StringUtil() {
	}



	//-----------------------------------------------------------------------
	//- methods
	//-----------------------------------------------------------------------

	/**
	 * 文字列内容がない場合、空文字列を返します。
	 *
	 * @param src 処理対象文字列
	 * @return 処理後の文字列
	 */
	public static String emptyToZeroString(String src) {
		return emptyToDefault(src, "");
	}

	/**
	 * 文字列内容がない場合、デフォルト文字列を返します。
	 *
	 * @param src 処理対象文字列
	 * @param defaultString 内容がない場合のデフォルト文字列
	 * @return 処理後の文字列
	 */
	public static String emptyToDefault(String src, String defaultString) {
		return isEmpty(src)? defaultString: src;
	}

	/**
	 * 文字列内容がない場合、nullを返します。
	 *
	 * @param src 処理対象文字列
	 * @return 処理後の文字列
	 */
	public static String emptyToNull(String src) {
		return emptyToDefault(src, null);
	}

	/**
	 * 文字列内容が存在するかチェックします。
	 *
	 * @param src チェック対象文字列
	 * @return 文字列内容が無い（nullまたは空文字）場合true
	 */
	public static boolean isEmpty(String src) {
		return src == null || src.equals("");
	}

	/**
	 * 文字列をtrim後、内容が存在するかチェックします。
	 *
	 * @param src チェック対象文字列
	 * @return trim後の文字列内容が無い（nullまたは空文字）場合true
	 */
	public static boolean isTrimEmpty(String src) {
		return isEmpty(trim(src));
	}

	/**
	 * 左からもしくは右から続く指定した文字を取り除きます。
	 *
	 * @param src 処理対象文字列
	 * @return 処理後の文字列
	 */
	public static String trim(String src) {
		return trim(src, WHITE_SPACES);
	}

	/**
	 * 左からもしくは右から続く指定した文字を取り除きます。
	 *
	 * @param src 処理対象文字列
	 * @param ch 取り除く文字
	 * @return 処理後の文字列
	 */
	public static String trim(String src, char ch) {
		if (src == null) return null;
		return rightTrim(leftTrim(src, ch), ch);
	}

	/**
	 * 左からもしくは右から続く指定した文字を取り除きます。
	 *
	 * @param src 処理対象文字列
	 * @param ch 取り除く文字
	 * @return 処理後の文字列
	 */
	public static String trim(String src, char[] chars) {
		if (src == null) return null;
		if (chars == null) return null;
		return rightTrim(leftTrim(src, chars), chars);
	}

	/**
	 * 左から続く空白（半角/全角共に）を取り除きます。
	 *
	 * @param src 処理対象文字列
	 * @return 処理後の文字列
	 */
	public static String leftTrim(String src) {
		return leftTrim(src, WHITE_SPACES);
	}

	/**
	 * 左から続く指定した文字を取り除きます。
	 *
	 * @param src 処理対象文字列
	 * @param ch 取り除く文字
	 * @return 処理後の文字列
	 */
	public static String leftTrim(String src, char ch) {
		return leftTrim(src, new char[]{ch});
	}

	/**
	 * 左から続く指定した文字を取り除きます。
	 *
	 * @param src 処理対象文字列
	 * @param chars 取り除く文字
	 * @return 処理後の文字列
	 */
	public static String leftTrim(String src, char[] chars) {
		if (src == null) return null;
		if (chars == null) return src;
		int i = 0;
		for (i = 0; i < src.length(); i++) {
			boolean hit = false;
			for (int j = 0; j < chars.length; j++) {
				char ch = chars[j];
				if (src.charAt(i) == ch) {
					hit = true;
					break;
				}
			}
			if (!hit) break;
		}
		return src.substring(i);
	}

	/**
	 * 右から続く空白（半角/全角共に）を取り除きます。
	 *
	 * @param src 処理対象文字列
	 * @return 処理後の文字列
	 */
	public static String rightTrim(String src) {
		return rightTrim(src, WHITE_SPACES);
	}

	/**
	 * 右から続く、指定した文字を取り除きます。
	 *
	 * @param src 処理対象文字列
	 * @param ch 取り除く文字
	 * @return 処理後の文字列
	 */
	public static String rightTrim(String src, char ch) {
		if (src == null) return null;
		int i = 0;
		for (i = src.length() - 1; i >= 0; i--) {
			if (src.charAt(i) != ch) break;
		}
		return src.substring(0, i + 1);
	}

	/**
	 * 右から続く、指定した文字を取り除きます。
	 *
	 * @param src 処理対象文字列
	 * @param chars 取り除く文字
	 * @return 処理後の文字列
	 */
	public static String rightTrim(String src, char[] chars) {
		if (src == null) return null;
		if (chars == null) return src;
		int i = 0;
		for (i = src.length() - 1; i >= 0; i--) {
			boolean hit = false;
			for (int j = 0; j < chars.length; j++) {
				char ch = chars[j];
				if (src.charAt(i) == ch) {
					hit = true;
					break;
				}
			}
			if (!hit) break;
		}
		return src.substring(0, i + 1);
	}

	/**
	 * 対象文字列が指定した文字数になるまで、文字を足しこみます。
	 * ただし文字は対象文字列の左に足しこんでいきます。
	 *
	 * @param src 対象文字列
	 * @param len 調節したい文字数
	 * @param ch 追加したい文字
	 * @return 加工した文字列
	 */
	public static String leftPad(long src, int len, char ch) {
		return leftPad(Long.toString(src), len, ch);
	}
	public static String leftPad(String src, int len, char ch) {
		if (src == null) return null;
		int nowLen = getByteLength(src);
		if (nowLen >= len) return src;
		len -= nowLen;;
		int chLen = getByteLength((Character.valueOf(ch)).toString());

		StringBuffer buf = new StringBuffer();
		for (int i = 0; i < len; ) {
			buf.append(ch);
			i += chLen;
		}
		return buf.append(src).toString();
	}

	/**
	 * 対象文字列が指定した文字数になるまで、文字を足しこみます。
	 * ただし文字は対象文字列の右に足しこんでいきます。
	 *
	 * @param src 対象文字列
	 * @param len 調節したい文字数
	 * @param ch 追加したい文字
	 * @return 加工した文字列
	 */
	public static String rightPad(long src, int len, char ch) {
		return rightPad(Long.toString(src), len, ch);
	}
	public static String rightPad(String src, int len, char ch) {
		if (src == null) return null;
		int nowLen = getByteLength(src);
		if (nowLen >= len) return src;
		len -= nowLen;
		int chLen = getByteLength((Character.valueOf(ch)).toString());

		StringBuffer buf = new StringBuffer(src);
		for (int i = 0; i < len; ) {
			buf.append(ch);
			i += chLen;
		}
		return buf.toString();
	}

	/**
	 * 文字列中にカウントしたい文字が何文字含まれているかをカウントします。
	 *
	 * @param src 処理対象文字列
	 * @param target カウントしたい文字
	 * @return 文字列中に含まれる指定文字の文字数
	 */
	public static int countCharacter(String src, char target) {
		int count = 0;
		for (int i = 0; i < src.length(); i++) {
			if (src.charAt(i) == target) count++;
		}

		return count;
	}

	/**
	 * 対象文字列のByte数を取得します。
	 * Ascii文字を1Byte、それ以外を2Byteと見します。
	 *
	 * @param src 対象文字列
	 * @return Byte数
	 */
	public static int getByteLength(String src) {
		int byteLen = 0;
		int len = src.length();
		for (int i = 0; i < len; i++) {
			byteLen += (CharChecker.isAscii(src.substring(i, i + 1)))? 1: 2;
		}
		return byteLen;
	}

	/**
	 * 対象文字列中から指定Byte数までの文字列を取得します。
	 * Ascii文字を1Byte、それ以外を2Byteと見なします。
	 *
	 * @param src 対象文字列
	 * @param limit 最大byte数
	 * @return 指定Byte数までの文字列
	 */
	public static String getLimitString(String src, int limit) {
		if (src == null) return null;

		int len = 0;
		int i = 0;
		for ( ; i < src.length(); i++) {
			char c = src.charAt(i);
			int b = getByteLength(String.valueOf(c));
			len += b;
			if (len > limit) break;
		}
		return src.substring(0, i);
	}

	/**
	 * 文字列配列を結合します。
	 *
	 * @param strs 結合対象文字列配列
	 * @return 結合された文字列
	 */
	public static String concat(final String[] strs) {
		return concat(strs, null);
	}
	/**
	 * 文字列コレクションを結合します。<br/>
	 * 文字列間には区切り文字を挿入します。<br/>
	 *
	 * @param strs 結合対象文字列コレクション
	 * @return 結合された文字列
	 */
	public static String concat(final Collection<?> strs) {
		return concat(strs, null);
	}

	/**
	 * 文字列配列を結合します。<br/>
	 * 文字列間には区切り文字を挿入します。<br/>
	 *
	 * @param strs 結合対象文字列配列
	 * @param delimita 区切り文字
	 * @return 結合された文字列
	 */
	public static String concat(final String[] strs, final String delimita) {
		return concat(Arrays.asList(strs), delimita);
	}

	/**
	 * 文字列コレクションを結合します。<br/>
	 * 文字列間には区切り文字を挿入します。<br/>
	 *
	 * @param strs 結合対象文字列コレクション
	 * @param delimita 区切り文字
	 * @return 結合された文字列
	 */
	public static String concat(final Collection<?> strs, final String delimita) {
		if (strs == null) return null;

		StringBuffer buf = new StringBuffer();
		int i = 0;
		for (Iterator<?> ite = strs.iterator(); ite.hasNext(); ) {
			Object obj = ite.next();
			String str = (obj == null)? null: obj.toString();

			buf.append(str);
			if (i == strs.size() - 1) break;
			if (delimita != null) buf.append(delimita);
			i++;
		}
		return buf.toString();
	}

	/**
	 * クォートをはずす。
	 *
	 * @param src 処理対象文字列
	 * @return 処理された文字列
	 */
	public static String unquate(String src) {
		String dst = null;
		dst = unquate(src, '"');
		if (!src.equals(dst)) return dst;
		dst = unquate(src, '\'');
		return dst;
	}

	/**
	 * クォートをはずす。
	 *
	 * @param src 処理対象文字列
	 * @param quate クォート文字
	 * @return 処理された文字列
	 */
	public static String unquate(String src, char quate) {
		if (src.charAt(0) != quate || src.charAt(src.length() - 1) != quate) return src;
		return src.substring(1, src.length() - 1);
	}

	/**
	 * 文字列を区切り文字で分割する。<br/>
	 * また、引用符で内に存在する区切り文字は無視する。<br/>
	 *
	 * @param str 分割する文字列
	 * @param delim 区切り文字
	 * @param quote 引用符
	 * @return	指定された文字列を指定された区切り文字に
	 * 				一致する位置で分割して計算された文字列の配列
	 * @throws	NoSuchElementException 不正に引用符が使用されていた場合
	 */
	public static String[] split(String str, final String delim, final String quote)
		throws NoSuchElementException {

		final int st = str.indexOf(quote);
		final int ed = str.lastIndexOf(quote);
		if (st < 0 || st >= ed) return str.split(delim);

		ArrayList<String> list = new ArrayList<String>();

		int current = 0, d, q;
		int onQuoted = 0;
		final int len = str.length();

		while (current < len) {
			d = str.indexOf(delim, current);
			q = str.indexOf(quote, current);

			if (onQuoted == 1) {
				if (q < 0) throw new NoSuchElementException("illegal quotation [" + str + "]");

				list.add(str.substring(current, q));
				current     = q + 1;
				onQuoted    = 2;
				continue;
			}

			if (d < 0 && q < 0) {
				String sub = str.substring(current, len).trim();
				if (sub.length() > 0) {
					if (onQuoted == 2) {
						throw new NoSuchElementException("illegal quotation [" + str + "]");
					}
					if (current < len) {
						list.add(sub);
					}
				}
				break;
			}

			if (d >= 0) {
				if (q < 0 || d < q) {
					if (onQuoted == 2) {
						onQuoted = 0;
					} else {
						list.add(str.substring(current, d).trim());
					}
					current = d + 1;
					continue;
				}
			}

			if (q >= 0) {
				current = q + 1;
				onQuoted = 1;
			}
		}

		String[] tokens = new String[list.size()];
		for (int i = 0; i < tokens.length; i++) {
			tokens[i] = (String)list.get(i);
		}
		return tokens;
	}


//	/** 変換用の全角数字 */
//	private static final char[] ZENKAKU_TABLE = {
//		'０', '１', '２', '３', '４', '５', '６', '７', '８', '９'
//	};
	private static final char[] HANKAKU_TABLE = {
		'0', '1', '2', '3', '4', '5', '6', '7', '8', '9'
	};
	private static final char ZENKAKU_DOT = '．';
	private static final char HANKAKU_DOT = '.';

	public static String toString(Object array) {
		StringBuilder sb = new StringBuilder();
		for(int i = 0; i < Array.getLength(array); i++) {
			if (i != 0)
				sb.append(",");
			Object o = Array.get(array, i);
			sb.append(o);
		}
		return sb.toString();
	}

	public static float[] parseFloat(String s) {
		StringTokenizer st = new StringTokenizer(s, ",");
		float[] res = new float[st.countTokens()];
		int pos = 0;
		while (st.hasMoreElements()) {
			try {
				res[pos] = Float.parseFloat(st.nextToken());
			} catch (NumberFormatException e) {
				res[pos] = 0;
			}
			pos++;
		}
		return res;
	}

	public static double[] parseDouble(String s){
		StringTokenizer st = new StringTokenizer(s, ",");
		double[] res = new double[st.countTokens()];
		int pos = 0;
		while (st.hasMoreElements()) {
			try {
				res[pos] = Double.parseDouble(st.nextToken());
			} catch (NumberFormatException e) {
				res[pos] = 0;
			}
			pos++;
		}
		return res;
	}

	/**
	 * @param s
	 * @return
	 * @since 2010/07/15
	 * @version 2010/07/15
	 */
	public static String toAsciiNumber(String s) {
		if(s==null)
			return null;
		char[] c = new char[s.length()];
		for (int i = 0; i < s.length(); i++) {
			char cc = s.charAt(i);
			if (cc >= '０' && cc <= '９') {
				c[i] = HANKAKU_TABLE[cc - '０'];
			}else if(cc==ZENKAKU_DOT){
				c[i] = HANKAKU_DOT;
			} else
				c[i] = cc;
		}
		return new String(c);
	}

} // end-class
