package jp.kitec.lib.util;

import java.io.Serializable;
import java.util.AbstractList;
import java.util.Iterator;
import java.util.List;


/**
 * 要素へのアクセスが、昇順／降順の双方向に可能な循環リスト。<BR>
 * また、すべての要素は、輪となって繋がっている。<BR>
 * Itemを直接操作する場合、整合性チェックのためO(n)のチェックが入るので注意が必要。<BR>
 * 比較する時にオブジェクトの同一性ではなくリファレンスの同一性を使用する。<BR>
 * 
 * visibility public<BR>
 * package    jp.kitec.lib.util<BR>
 * name       RefList<BR>
 * implements Serializable<BR>
 * 
 * @since 　2002/12/20
 * @author　fujita
 * @version 2002/12/20
 * TODO implement java.util.List<E>
 * 
 * Copyright (c) 2002 KITec Inc,.. All rights reserved.
 */
public class RefList<E> extends AbstractList<E> implements KCollection<E>, Serializable, Iterable<E>, List<E> {
	
	/////////////////////////////////////////////////////////////////////////
	// フィールド
	
	/** リスト上の先頭となるアイテム */
	private Item<E> head;

	/** 登録されているアイテム数 */
	private int regCount;


	/**
	 * アイテム数 0 の、空の RefList を構築する。
	 * 
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public RefList() {
		head = null;
		regCount = 0;
	}

	/**
	 * デバッグ用。
	 * 
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public RefList(String s) {
		head = null;
		regCount = 0;
	}

	/**
	 * 指定されたアイテムを、この RefList の先頭を指す要素として設定する。
	 * 指定されたアイテムが、この RefList の要素に含まれない場合、
	 * 先頭へは設定されない。
	 * 
	 * @param it 先頭の要素として設定するアイテム 
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public void setHead(Item<E> it) {
		if (!containsItem(it))
			return;
		head = it;
	}

	/**
	 * この RefList の先頭の要素を返す。
	 *
	 * @return 先頭の要素 
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public E firstElement() {
		if (regCount != 0)
			return head.getObject();
		return null;
	}

	/**
	 * この RefList の最後の要素を返す。
	 *
	 * @return 最後の要素 
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public E lastElement() {
		if (regCount != 0)
			return head.prevItem.getObject();
		return null;
	}

	/**
	 * 現在の先頭の要素を基点として、
	 * 指定されたインデックス位置の要素を返す。
	 * 
	 * @param c インデックス
	 * @return 指定されたインデックス位置にある要素
	 * @see #itemAt(int)
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public E elementAt(int c) {
		return itemAt(c).getObject();
	}
	
	/**
	 * 現在の先頭の要素を基点として、
	 * 指定されたインデックス位置のアイテムを返す。
	 * 
	 * @param c インデックス
	 * @return 指定されたインデックス位置にあるアイテム
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public Item<E> itemAt(int c) {
		Item<E> it = head;
		if (it == null)
			throw new IndexOutOfBoundsException();

		if (c >= 0) {
			for (int i = 0; i < c; i++)
				it = it.nextItem;
		} else {
			for (int i = 0; i > c; i--)
				it = it.prevItem;
		}
		return it;
	}

	/**
	 * 指定されたデータを持つ要素を返す。
	 * 
	 * @param o 要素
	 * @return 指定された要素を持つアイテム
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public Item<E> itemAt(Object o) {
		Item<E> it = head;
		for (int i = 0; i < regCount; i++) {
			if (it.getObject() == o) {
				return it;
			}
			it = it.nextItem;
		}
		return null;
	}


	/**
	 * 現在の RefList の要素数を返す。
	 * 
	 * @return  RefList の要素数
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public int size() {
		return regCount;
	}

	/**
	 * 指定された要素を RefList 最後に追加し、サイズを 1 つ増やす。
	 * 
	 * @param   o 新たに追加する要素
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public	boolean addElement(E o) {
		if (o == null)
			return false;
		if (head == null)
			insertElementAt(o, head);
		else
			insertElementAt(o, head.prevItem);

		return true;
	}
	
	/**
	 * この RefList が保持するすべての要素を削除する。
	 * 削除後、 RefList のサイズは 0 となる。
	 * 
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public void removeAllElements() {
//		{
//			Item item = head;
//			while (true) {
//				Item next = item.nextItem;
//				if (item.prevItem != null)
//					item.prevItem.nextItem = null;
//				if (item.nextItem != null)
//					item.nextItem.prevItem = null;
//				
//				item.prevItem = item.nextItem = null;
//				if (next == null)
//					break;
//			}
//			if (true)
//				return;
//		}
		
		
		int c = regCount;
		for (int i = 0; i < c; i++)
			removeItemAt(head);
	}

	/**
	 * 指定された要素を RefList から検索し、最初に見つかった要素を削除する。
	 * 
	 * @param	o	削除する要素
	 * @return	該当する要素が見つからなければ、null。
	 *			見つかれば削除された要素の前のアイテム
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public Item<E> removeElement(Object o) {
		Item<E> delitem = null;

		Item<E> it = head;
		for (int i = 0; i < regCount; i++) {
			if (it.getObject() == o) {
				delitem = it;
				break;
			}
			it = it.nextItem;
		}

		if (delitem != null)
			return removeItemAt(delitem);
		return null;
	}

	/**
	 * 指定されたアイテムを RefList から検索し、最初に見つかったアイテムを削除する。
	 * 
	 * @param	ref		削除するアイテム
	 * @return	該当するアイテムが見つからなければnull
	 *			見つかれば削除されたアイテムの前のアイテム
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public Item<E> removeItemAt(Item<E> ref) {
		Item<E> now;

		if (ref == null || head == null || ref.nextItem == null)
			return null;

		if (regCount > 1) {
			if (ref == head)
				head = ref.nextItem;
			ref.prevItem.nextItem = ref.nextItem;
			now = ref.nextItem.prevItem = ref.prevItem;
			//消去
			ref.nextItem = ref.prevItem = null;
			ref = null;
		} else
			now = head = null;

		regCount--;

		return (now);
	}

	/**
	 * 指定されたオブジェクトが要素かどうかを判定する。
	 * 
	 * @param o	オブジェクト
	 * @return 指定されたオブジェクトが equals メソッドによって
	 * 			RefList 内の要素と同じと確認された場合にだけ true、
	 * 			そうでない場合は false
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public boolean contains (Object o) {
		Item<E> it = head;
		for (int i = 0; i < regCount; i++) {
			if (it.getObject() == o)
				return true;
			it = it.nextItem;
		}
		return false;
	}

	/**
	 * 指定されたアイテムが要素かどうかを判定する。
	 * 
	 * @param ptr	アイテム
	 * @return 指定されたアイテムが equals メソッドによって
	 * 			RefList 内の要素と同じと確認された場合にだけ true、
	 * 			そうでない場合は false
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	private boolean containsItem (Item<E> ptr) {
		Item<E> it = head;
		for (int i = 0; i < regCount; i++) {
			if (ptr == it)
				return true;
			it = it.nextItem;
		}
		return false;
	}

	/**
	 * 指定されたオブジェクトを新たな要素として
	 * RefList の指定された位置(指定されたindexの直後)へ挿入する。
	 * 
	 * @param	ptr		挿入される要素
	 * @param	index	新しい要素を挿入する位置
	 * @return	要素が追加された場合、追加されたアイテム。
	 * 			インデックスが不正な場合など、
	 * 			追加されなかった場合は、null
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public Item<E> insertElementAt(E data, int index) {
		if (data == null)
			return null;
		if (index < 0 || index >= regCount)
			return null;
		Item<E> it = head;
		for (int i = 0; i < regCount; i++) {
			if (i == index)
				break;
			it = it.nextItem;
		}
		return insertElementAt(data, it);
	}

	/**
	 * 指定されたオブジェクトを新たな要素として
	 * RefList の指定されたアイテムの直後に挿入する。
	 * TODO:注意！！！　insertの順番が間違っている可能性あり。
	 * 
	 * @param	data	挿入される要素
	 * @param	ptr		新しい要素を挿入する直前のアイテム
	 * @return	要素が追加された場合、追加されたアイテム。
	 * 			インデックスが不正な場合など、
	 * 			追加されなかった場合は、null
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public Item<E> insertElementAt(E data, Item<E> ptr) {
		if ((ptr == null && head != null) || data == null)
			return null;

		if (!containsItem(ptr) && head != null)
			return null;

		Item<E> newItem = new Item<E>(data);

		if (ptr == null) {
			newItem.nextItem = newItem.prevItem = newItem;
			head = newItem;
		} else {
//			newItem.prevItem = ptr.prevItem;
//			newItem.nextItem = ptr;
//			ptr.nextItem.prevItem = newItem;
//			ptr.nextItem = newItem;
			newItem.prevItem = ptr;
			newItem.nextItem = ptr.nextItem;
			ptr.nextItem.prevItem = newItem;
			ptr.nextItem = newItem;
		}
		regCount++;

		return newItem;
	}

	/**
	 * この RefList の要素を共有する新たな RefList を構築し、返す。
	 * 
	 * @return	要素を共有する新たな RefLsit
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public RefList<E> getCopy() {
		RefList<E> tmp = new RefList<E>();
		for (int i = 0; i < this.size(); i++)
			tmp.addElement(this.elementAt(i));
		return tmp;
	}

	/**
	 * 指定された 2 つのアイテムが示す要素を入れ替える。
	 * 
	 * @param i1 要素入れ替えるアイテム 1
	 * @param i2 要素入れ替えるアイテム 2
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public <T> void swap(Item<T> i1, Item<T> i2) {
		T o = i1.getObject();
		i1.setObject(i2.getObject());
		i2.setObject(o);
	}

	/**
	 * it1 ～ it2 の範囲に含まれるすべての要素をこの RefList から削除する。
	 * 
	 * @param i1	削除する最初の要素
	 * @param i2	削除する最後の要素
	 * @param dir	削除を行う方向。true の場合、i1 から順（後方）に i2 まで
	 * 				の要素を削除する。false の場合、i1 から逆順（前方）に要素
	 * 				を削除する。
	 * @return 指定された範囲の要素が削除された場合、true。指定された要素が
	 * 			RefList に含まれない場合など削除がされなかった場合、false
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public boolean deleteAtoB(Item<E> it1, Item<E> it2, boolean dir) {
		Item<E> start = null, end = null;
		Item<E> it = head;
		if (it1 == it2 && it1 != null) {
			removeItemAt(it1);
			return true;
		}
		for (int i = 0; i < regCount; i++) {
			if (it == it1)
				start = it;
			else if (it == it2)
				end = it;
			if (start != null && end != null)
				break;
			it = it.nextItem;
		}
		if (start == null || end == null)
			return false;

		int count = regCount;
		it = start;
		for (int i = 0; i < count; i++) {
			Item<E> itnext = dir ? it.nextItem : it.prevItem;
			removeItemAt(it);
			if (it == end)
				break;
			it = itnext;
		}
		return true;
	}

	/**
	 * この RefList が保持する要素の順番を反転する。
	 * 例）	処理前：[A][B][C][D][E]
	 * 		処理後：[A][E][D][C][B]
	 * 
	 * @since   2002/12/20
	 * @author  Kawae
	 * @version 2002/12/20
	 */
	public void flip() {
		Item<E> n, ncur, nend;
		ncur = head;
		nend = head.nextItem;
		do {
			n = head.prevItem;
			removeItemAt(n);
			insertElementAt(n.getObject(), ncur);
			ncur = ncur.nextItem;
		} while (n.getObject() != nend.getObject());
	}
	

	@Override
	public int hashCode() {
		int hashCode = 1;
		Item<E> it = head;
		for (int i = 0; i < regCount; i++) {
			E obj = it.getObject();
			hashCode = 31*hashCode + (obj==null ? 0 : obj.hashCode());
			it = it.nextItem;
		}
		return hashCode;
	}

	@Override
	@SuppressWarnings("rawtypes")
	public boolean equals(Object o) {
		if (o == this)
			return true;
		if (!(o instanceof RefList))
			return false;
		
		RefList rl2 = (RefList)o;
		if (regCount != rl2.regCount)
			return false;
		
		Item it1 = head;
		Item it2 = rl2.head;
		for (int i = 0; i < regCount; i++) {
			Object o1 = it1.getObject();
			Object o2 = it2.getObject();

			if (!(o1==null ? o2==null : o1.equals(o2)))
				return false;
			
			it1 = it1.nextItem;
			it2 = it2.nextItem;
		}

		return true;
	}

	@Override
	public Iterator<E> iterator() {
		return new Iterator<E>() {
			Item<E> current = head;
			int n = 0;
			@Override
			public boolean hasNext() {
				return n < regCount;
			}

			@Override
			public E next() {
				E o = current.getObject();
				current = current.nextItem;
				n++;
				return o;
			}

			@Override
			public void remove() {
				throw new UnsupportedOperationException();
			}
			
		};
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public E get(int index) {
		return elementAt(index);
	}

	/**
	 * {@inheritDoc}
	 */
	public E set(int index, E element) {
		Item<E> item = itemAt(index);
		E org = item.getObject();
		item.setObject(element);
		return org;
	}

	/**
	 * {@inheritDoc}
	 */
	public void add(int index, E element) {
		insertElementAt(element, index);
	}

	/**
	 * {@inheritDoc}
	 */
	public E remove(int index) {
		Item<E> removeItem = itemAt(index);
		removeItemAt(removeItem);
		return removeItem.getObject();
	}

//	@Override
//	public Object[] toArray() {
//		Object[] ar = new Object[regCount];
//
//		Item<E> it = head;
//		for (int i = 0; i < regCount; i++) {
//			ar[i] = it.getObject();
//			it = it.nextItem;
//		}
//	
//		return null;
//	}
}
