/*
 * Copyright (c) 2009-2010 KITec Inc,.. All rights reserved.
 */
package jp.kitec.lib.util;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;



/**
 * メッセージダイジェストユーティティ
 *
 * @author $Author$
 * @version $Revision$ $Date::                           $
 */
public class DigestUtil {

	/**
	 * アルゴリズム「MD2」を使用して、メッセージダイジェストを計算する。
	 *
	 * @param src メッセージ
	 * @return メッセージダイジェスト
	 */
	public static String md2(String src) {
		return digestNoException("MD2", src);
	}

	/**
	 * アルゴリズム「MD5」を使用して、メッセージダイジェストを計算する。
	 *
	 * @param src メッセージ
	 * @return メッセージダイジェスト
	 */
	public static String md5(String src) {
		return digestNoException("MD5", src);
	}

	/**
	 * アルゴリズム「SHA-1」を使用して、メッセージダイジェストを計算する。
	 *
	 * @param src メッセージ
	 * @return メッセージダイジェスト
	 */
	public static String sha1(String src) {
		return digestNoException("SHA-1", src);
	}

	/**
	 * アルゴリズム「SHA-256」を使用して、メッセージダイジェストを計算する。
	 *
	 * @param src メッセージ
	 * @return メッセージダイジェスト
	 */
	public static String sha256(String src) {
		return digestNoException("SHA-256", src);
	}

	/**
	 * アルゴリズム「SHA-384」を使用して、メッセージダイジェストを計算する。
	 *
	 * @param src メッセージ
	 * @return メッセージダイジェスト
	 */
	public static String sha384(String src) {
		return digestNoException("SHA-384", src);
	}

	/**
	 * アルゴリズム「SHA-512」を使用して、メッセージダイジェストを計算する。
	 *
	 * @param src メッセージ
	 * @return メッセージダイジェスト
	 */
	public static String sha512(String src) {
		return digestNoException("SHA-512", src);
	}

	/**
	 * 任意のアルゴリズムを使用して、メッセージダイジェストを計算する。<br/>
	 * 使用できないアルゴリズム名を指定した場合、nullを返す。<br/>
	 *
	 * @param algorithm アルゴリズム名(MD2/MD5/SHA-1/SHA-256/SHA-384/SHA-512)
	 * @param src メッセージ
	 * @return メッセージダイジェスト
	 */
	private static String digestNoException(String algorithm, String src) {
		String result = null;
		try {
			result = digest(algorithm, src);
		} catch (NoSuchAlgorithmException e) {
		}
		return result;
	}

	/**
	 * 任意のアルゴリズムを使用して、メッセージダイジェストを計算する。<br/>
	 * 使用できないアルゴリズム名を指定した場合、{@link NoSuchAlgorithmException}をスローする。<br/>
	 *
	 * @param algorithm アルゴリズム名(MD2/MD5/SHA-1/SHA-256/SHA-384/SHA-512)
	 * @param src メッセージ
	 * @return メッセージダイジェスト
	 * @throws NoSuchAlgorithmException
	 */
	public static String digest(String algorithm, String src) throws NoSuchAlgorithmException {
		byte[] md5Bytes = null;
		MessageDigest digest = MessageDigest.getInstance(algorithm);
		md5Bytes = digest.digest(src.getBytes());
		return toHexString(md5Bytes);
	}

	private static String toHexString(byte[] bytes) {
		StringBuffer buf = new StringBuffer(bytes.length * 2);
		for (int i = 0; i < bytes.length; i++) {
			String b = Integer.toHexString(bytes[i] & 0xff);
			if (b.length() == 1) buf.append("0");
			buf.append(b);
		}
		return buf.toString();
	}



} // end-class
