/*
 * Copyright (c) 2008 KITec Inc,.. All rights reserved.
 */
package jp.kitec.lib.util;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;



/**
 * Framework Collectionユーティリティクラス
 *
 * @author $Author$
 * @version $Revision$ $Date::                           $
 */
public class CollectionUtil {

	//------------------------------------------------------------------
	//- constructors
	//------------------------------------------------------------------

	/**
	 * コンストラクタ
	 */
	protected CollectionUtil() {
	}



	//------------------------------------------------------------------
	//- methods
	//------------------------------------------------------------------

	/**
	 * コレクション中の位置を取得する。<br/>
	 * コレクションに含まれていない場合は-1を返す。<br/>
	 *
	 * @param objects コレクション
	 * @param target 対象オブジェクト
	 * @return コレクション中の位置
	 */
	public static int indexOf(final Collection<?> objects, final Object target) {
		int index = -1;
		if (objects == null) return index;
		if (target == null) return index;

		int i = 0;
		for (Object object: objects) {
			if (object == target) {
				index = i;
				break;
			}
			i++;
		}

		return index;
	}

	/**
	 * コレクション内の指定された位置にある要素を返す。<br/>
	 *
	 * @param objects コレクション
	 * @param index 指定位置
	 * @return 要素
	 */
	public static <T> T get(final Collection<T> objects, final int index) {
		if (objects == null) return null;
		if (index < 0 || objects.size() <= index) return null;

		T result = null;
		int count = 0;
		for (Iterator<T> ite = objects.iterator(); ite.hasNext(); ) {
			T object = ite.next();
			if (count == index) {
				result = object;
				break;
			}
			count++;
		}

		return result;
	}

	/**
	 * 左側に存在する要素を取得する。
	 *
	 * @param objects コレクション
	 * @param baseObject 基点となる要素
	 * @return 左側の要素
	 */
	public static <T> T getLeftNextObject(final Collection<T> objects, final T baseObject) {
		if (objects == null) return null;
		if (baseObject == null) return null;

		int index = indexOf(objects, baseObject);
		if (index == -1) return null;
		if (index == 0) return null;

		return get(objects, index - 1);
	}

	/**
	 * 右側に存在する要素を取得する。
	 *
	 * @param objects コレクション
	 * @param baseObject 基点となる要素
	 * @return 右側の要素
	 */
	public static <T> T getRightNextObject(final Collection<T> objects, final T baseObject) {
		if (objects == null) return null;
		if (baseObject == null) return null;

		int index = indexOf(objects, baseObject);
		if (index == -1) return null;
		if (index == objects.size() - 1) return null;

		return get(objects, index + 1);
	}

	/**
	 * コレクション最初の要素を取得する。
	 *
	 * @param objects コレクション
	 * @return 最初の要素
	 */
	public static <T> T getFirstObject(final Collection<T> objects) {
		if (objects == null) return null;
		return objects.iterator().next();
	}

	/**
	 * コレクション最後の要素を取得する。
	 *
	 * @param objects コレクション
	 * @return 最後の要素
	 */
	public static <T> T getLastObject(final Collection<T> objects) {
		if (objects == null) return null;
		return get(objects, objects.size() - 1);
	}

	/**
	 * コレクションに配列要素を追加する。
	 *
	 * @param objects コレクション
	 * @param objs 配列
	 */
	public static <T, U extends T> void addAll(final Collection<T> objects, U[] objs) {
		if (objects == null) return;
		for (U obj: objs) objects.add(obj);
	}

	/**
	 * コレクションをソートする。
	 *
	 * @param <T> 任意の型
	 * @param objects コレクション
	 * @param comp {@link Comparator}
	 */
	public static <T> void sort(Collection<T> objects, Comparator<T> comp) {
		List<T> tmpList = new LinkedList<T>(objects);
		Collections.sort(tmpList, comp);
		objects.clear();
		objects.addAll(tmpList);
	}

	/**
	 * コレクションから指定クラスを収集する。
	 *
	 * @param objs コレクション
	 * @param findClazz {@link Class}
	 * @return コレクション
	 */
	@SuppressWarnings("unchecked")
	public static <T, U> Collection<T> collect(final Collection<U> objs, final Class<T> findClazz) {
		Set<T> result = new SetNotNullDecorator<T>(new LinkedHashSet<T>());

		for (Object obj: objs) {
			if (findClazz.isInstance(obj)) result.add((T)obj);
		}

		return result;
	}

	/**
	 * Collectionから指定値を取り除く。
	 *
	 * @param src filter対象Collection
	 * @param dst filter結果Collection
	 * @param filterValues 取り除く対象
	 * @return filter結果Collection
	 */
	public static <V, T extends Collection<V>> T valueFilter(T src, T dst, Object[] filterValues) {
		for (V value: src) {
			boolean match = false;
			for (Object filterValue: filterValues) {
				if (value == filterValue
					|| (value != null && value.equals(filterValue))) {
					match = true;
					break;
				}
			}
			if (match) continue;

			dst.add(value);
		}
		return dst;
	}

	/**
	 * Mapから指定値を取り除く。
	 *
	 * @param src filter対象Map
	 * @param dst filter結果Map
	 * @param filterValues 取り除く対象
	 * @return filter結果Map
	 */
	public static <K, V, T extends Map<K, V>> T valueFilter(T src, T dst, Object[] filterValues) {
		for (Entry<K, V> entry: src.entrySet()) {
			boolean match = false;
			for (Object filterValue: filterValues) {
				if (entry.getValue() == filterValue
					|| (entry.getValue() != null && entry.getValue().equals(filterValue))) {
					match = true;
					break;
				}
			}
			if (match) continue;

			dst.put(entry.getKey(), entry.getValue());
		}
		return dst;
	}



} // end-class
