/*
 * Copyright (c) 2013-2014 KITec Inc,.. All rights reserved.
 */
package option.gad.core.util.csv;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import jp.kitec.lib.util.NameUtil;
import jp.kitec.lib.util.ReflectUtil;
import option.gad.core.dxo.TypeConvertUtil;
import option.gad.core.io.FileIOUtil;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;



/**
 * CSV構築クラス
 *
 * @author $Author: tanaka $
 * @version $Revision: 552 $ $Date:: 2010-06-20 19:06:19 +0900#$
 */
public class CsvBuilder {

	//------------------------------------------------------------------
	//- constants
	//------------------------------------------------------------------

	/** デフォルト・文字エンコーディング */
	protected static final String DEFAULT_CHARCTER_ENCODING = "MS932";



	//------------------------------------------------------------------
	//- fields
	//------------------------------------------------------------------

	/** ログ */
	private static final Log mLog = LogFactory.getLog(CsvBuilder.class);



	//------------------------------------------------------------------
	//- constructors
	//------------------------------------------------------------------

	/**
	 * コンストラクタ
	 */
	public CsvBuilder() {
	}



	//------------------------------------------------------------------
	//- methods
	//------------------------------------------------------------------

	/**
	 * CSVデータを構築する。
	 */
	public Csv build(String str, boolean head) {
		return build(str, head, DEFAULT_CHARCTER_ENCODING);
	}

	/**
	 * CSVデータを構築する。
	 */
	public Csv build(String str, boolean head, String characterEncoding) {
		return build(new ByteArrayInputStream(str.getBytes()), head, characterEncoding);
	}

	/**
	 * CSVデータを構築する。
	 */
	public Csv build(File file, boolean head) {
		return build(file, head, DEFAULT_CHARCTER_ENCODING);
	}

	/**
	 * CSVデータを構築する。
	 */
	public Csv build(File file, boolean head, String characterEncoding) {
		Csv csv = null;
		try {
			csv = build(FileIOUtil.openInReadingMode(file), head, characterEncoding);
		} catch (FileNotFoundException e) {
			mLog.info("", e);
		}
		return csv;
	}

	/**
	 * CSVデータを構築する。
	 */
	public Csv build(InputStream is, boolean head) {
		return build(is, head, DEFAULT_CHARCTER_ENCODING);
	}

	/**
	 * CSVデータを構築する。
	 */
	public Csv build(InputStream is, boolean head, String characterEncoding) {
		return build(is, head, false, characterEncoding);
	}

	/**
	 * CSVデータを構築する。<br/>
	 * このメソッドでは次の処理に対応できていない。<br/>
	 * ・セルを括っているダブルクォートの除去<br/>
	 * ・セル内のカンマの保持。（セル内のカンマで、セル分割を行ってしまう）<br/>
	 */
	public Csv build(InputStream is, boolean head, boolean correctCamelCase, String characterEncoding) {
		Csv csv = null;
		BufferedReader br = null;
		try {
			csv = new Csv();
			br = new BufferedReader(new InputStreamReader(is, characterEncoding));
			String line;
			while ((line = br.readLine()) != null) {
				if (line.startsWith("#")) continue;

				String[] cols = line.split(",", -1);
				if (head) {
					for (String col: cols) {
						if (col != null) {
							col = col.trim();
							if (correctCamelCase) col = NameUtil.convCamelCase(col);
						}
						csv.mColNameList.add(col);
					}
					head = false;
				} else {
					if (csv.mColNameList.isEmpty()) {
						for (int i = 0; i < cols.length; i++) {
							csv.mColNameList.add(Integer.toString(i));
						}
					}
					List<String> colList = new ArrayList<String>();
					int i = 0;
					for (String col: cols) {
						if (col != null) col = col.trim();
						if (i >= csv.mColNameList.size()) {
							mLog.info(line + " : " + csv.mColNameList.size() + " != " + i);
							continue;
						}
						colList.add(col);
					}
					csv.mRowList.add(colList);
				}
			}
		} catch (IOException e) {
			mLog.info("", e);
			throw new RuntimeException(e);
		} finally {
			FileIOUtil.close(br);
		}

		return csv;
	}

	/**
	 * CSVデータを構築する。
	 */
	public Csv build(List<?> rows) {
		Csv csv = new Csv();

		Collection<Field> fieldList = null;
		for (Object row: rows) {
			if (csv.mRowList.isEmpty()) {
				fieldList = ReflectUtil.getFields(row);
				for (Field field: fieldList) {
					String colName = NameUtil.removePrefix(field.getName());
					csv.mColNameList.add(colName);
				}
			}

			ArrayList<String> colList = new ArrayList<String>();
			for (Field field: fieldList) {
				Object value = ReflectUtil.getFieldValue(row, field);
				String convertedValue = TypeConvertUtil.convertString(value);
				colList.add(convertedValue);
			}
			csv.mRowList.add(colList);
		}

		return csv;
	}



} // end-class
