/*
 * Copyright (c) 2009-2013 KITec Inc,.. All rights reserved.
 */
package option.gad.core.util;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.NoSuchElementException;

import jp.kitec.lib.util.ClassInfo;
import jp.kitec.lib.util.ClassInfoManager;
import jp.kitec.lib.util.NameUtil;
import jp.kitec.lib.util.ReflectUtil;
import option.gad.core.dxo.TypeConvertUtil;



/**
 * 構造体ユーティリティ
 *
 * @author $Author$
 * @version $Revision$ $Date::                           $
 */
public class StructUtil {

	//------------------------------------------------------------------
	//- constructors
	//------------------------------------------------------------------

	/**
	 * コンストラクタ
	 */
	protected StructUtil() {
		super();
	}



	//------------------------------------------------------------------
	//- methods
	//------------------------------------------------------------------

	/**
	 * 構造体から文字列配列を生成する。
	 *
	 * @param struct 構造体
	 * @return 文字列配列
	 */
	public static <E> String[] createStrings(final E entity) {
		return createStrings(entity, null);
	}

	/**
	 * 構造体から文字列配列を生成する。
	 *
	 * @param struct 構造体
	 * @param annoClazz 処理対象アノテーション型
	 * @return 文字列配列
	 */
	public static <E, A extends Annotation> String[] createStrings(final E entity, final Class<A> annoClazz) {
		int fieldSize = 0;
		for (Field field: ReflectUtil.getFields(entity)) {
			if (annoClazz != null && field.getAnnotation(annoClazz) == null) continue;
			fieldSize++;
		}

		String[] strs = new String[fieldSize];
		settingStructToStrings(entity, strs, annoClazz);
		return strs;
	}

	/**
	 * 構造体を文字列配列に設定する。
	 *
	 * @param struct 構造体
	 * @param strs 文字列配列
	 * @param annoClazz 処理対象アノテーション型
	 */
	public static <E> void settingStructToStrings(final E struct, String[] strs) {
		settingStructToStrings(struct, strs, null);
	}

	/**
	 * 構造体を文字列配列に設定する。
	 *
	 * @param struct 構造体
	 * @param strs 文字列配列
	 * @param annoClazz 処理対象アノテーション型
	 */
	public static <E, A extends Annotation> void settingStructToStrings(final E struct, String[] strs, final Class<A> annoClazz) {
		settingStructToStrings(struct, ReflectUtil.getFields(struct.getClass()), strs, annoClazz);
	}

	/**
	 * 構造体を文字列配列に設定する。
	 *
	 * @param struct 構造体
	 * @param fields Fieldコレクション
	 * @param strs 文字列配列
	 * @param annoClazz 処理対象アノテーション型
	 */
	protected static <A extends Annotation>
			void settingStructToStrings(final Object struct, final Collection<Field> fields, final String[] strs, final Class<A> annoClazz) {
		Map<String, Object> map = new LinkedHashMap<String, Object>();

		int strCount = 0;
		for (Field field: fields) {
			if (annoClazz != null && field.getAnnotation(annoClazz) == null) continue;

			map.clear();
			if (settingStructToMap(struct, field, map)) {
				strs[strCount] = TypeConvertUtil.convert(String.class, map.values().iterator().next());
			}
			strCount++;
		}
	}

	/**
	 * 構造体からMapを生成する。
	 *
	 * @param struct 構造体
	 * @return Map
	 */
	public static <E> Map<String, Object> createMap(final E entity) {
		return createMap(entity, null);
	}

	/**
	 * 構造体からMapを生成する。
	 *
	 * @param struct 構造体
	 * @param annoClazz 処理対象アノテーション型
	 * @return Map
	 */
	public static <E, A extends Annotation> Map<String, Object> createMap(final E entity, final Class<A> annoClazz) {
		Map<String, Object> map = new LinkedHashMap<String, Object>();
		settingStructToMap(entity, map, annoClazz);
		return map;
	}

	/**
	 * 構造体をMapに設定する。
	 *
	 * @param struct 構造体
	 * @param map Map
	 * @param annoClazz 処理対象アノテーション型
	 */
	public static <E> void settingStructToMap(final E struct, Map<String, Object> map) {
		settingStructToMap(struct, map, null);
	}

	/**
	 * 構造体をMapに設定する。
	 *
	 * @param struct 構造体
	 * @param map Map
	 * @param annoClazz 処理対象アノテーション型
	 */
	public static <E, A extends Annotation> void settingStructToMap(final E struct, Map<String, Object> map, final Class<A> annoClazz) {
		settingStructToMap(struct, ReflectUtil.getFields(struct.getClass()), map, annoClazz);
	}

	/**
	 * 構造体を文字列配列に設定する。
	 *
	 * @param struct 構造体
	 * @param fields Fieldコレクション
	 * @param map Map
	 * @param annoClazz 処理対象アノテーション型
	 */
	protected static <A extends Annotation>
			void settingStructToMap(final Object struct, final Collection<Field> fields, final Map<String, Object> map, final Class<A> annoClazz) {
		for (Field field: fields) {
			if (annoClazz != null && field.getAnnotation(annoClazz) == null) continue;
			settingStructToMap(struct, field, map);
		}
	}

	/**
	 * 構造体をMapに設定する。
	 *
	 * @param struct 構造体
	 * @param field Field
	 * @param map Map
	 * @return 設定できた場合true。
	 */
	protected static boolean settingStructToMap(final Object struct, final Field field, final Map<String, Object> map) {
		Object value = ReflectUtil.getFieldValue(struct, field);
		if (value == null) return false;

		String fieldName = NameUtil.removePrefix(field.getName());
		map.put(fieldName, value);
		return true;
	}

	/**
	 * オブジェクト配列から構造体を生成する。<br/>
	 * オブジェクトと構造体フィールドの順序は一致している必要があります。<br/>
	 *
	 * @param clazz 構造体型
	 * @param objs オブジェクト配列
	 * @return 構造体
	 */
	public static <E> E createStruct(final Class<E> clazz, final Object[] objs) {
		return createStruct(clazz, objs, null);
	}

	/**
	 * オブジェクト配列から構造体を生成する。<br/>
	 * オブジェクトと構造体フィールドの順序は一致している必要があります。<br/>
	 *
	 * @param clazz 構造体型
	 * @param objs オブジェクトコレクション
	 * @param annoClazz 処理対象アノテーション型
	 * @return 構造体
	 */
	public static <E, A extends Annotation> E createStruct(final Class<E> clazz, final Object[] objs, final Class<A> annoClazz) {
		return createStruct(clazz, Arrays.asList(objs), annoClazz);
	}

	/**
	 * オブジェクトコレクションから構造体を生成する。<br/>
	 * オブジェクトと構造体フィールドの順序は一致している必要があります。<br/>
	 *
	 * @param clazz 構造体型
	 * @param objs オブジェクトコレクション
	 * @return 構造体
	 */
	public static <E, O extends Object> E createStruct(final Class<E> clazz, final Collection<O> objs) {
		return StructUtil.createStruct(clazz, objs, null);
	}

	/**
	 * オブジェクトコレクションから構造体を生成する。<br/>
	 * オブジェクトと構造体フィールドの順序は一致している必要があります。<br/>
	 *
	 * @param clazz 構造体型
	 * @param objs オブジェクトコレクション
	 * @param annoClazz 処理対象アノテーション型
	 * @return 構造体
	 */
	@SuppressWarnings("unchecked")
	public static <E, O extends Object, A extends Annotation>
			E createStruct(final Class<E> clazz, final Collection<O> objs, final Class<A> annoClazz) {
		ClassInfo structClazzInfo = ClassInfoManager.getInstance().getClassInfo(clazz);
		E struct = (E)ReflectUtil.newInstance(structClazzInfo.getType());
		clearStruct(struct, annoClazz);
		settingObjectsToStruct(objs, struct, annoClazz);
		return struct;
	}

	/**
	 * オブジェクトコレクションを構造体に設定する。<br/>
	 * オブジェクトと構造体フィールドの順序は一致している必要があります。<br/>
	 *
	 * @param objs オブジェクトコレクション
	 * @param struct 構造体
	 */
	public static <E, O extends Object> void settingObjectsToStruct(final Collection<O> objs, final E struct) {
		settingObjectsToStruct(objs, struct, null);
	}

	/**
	 * オブジェクトコレクションを構造体に設定する。<br/>
	 * オブジェクトと構造体フィールドの順序は一致している必要があります。<br/>
	 *
	 * @param objs オブジェクトコレクション
	 * @param struct 構造体
	 * @param annoClazz 処理対象アノテーション型
	 */
	public static <E, O extends Object, A extends Annotation>
			void settingObjectsToStruct(final Collection<O> objs, final E struct, final Class<A> annoClazz) {
		settingStringsToStruct(objs, struct, ReflectUtil.getFields(struct.getClass()), annoClazz);
	}

	/**
	 * オブジェクトコレクションを構造体に設定する。<br/>
	 * オブジェクトと構造体フィールドの順序は一致している必要があります。<br/>
	 *
	 * @param objs オブジェクトコレクション
	 * @param struct 構造体
	 * @param fields Fieldコレクション
	 * @param annoClazz 処理対象アノテーション型
	 */
	protected static <E, O extends Object, A extends Annotation>
			void settingStringsToStruct(final Collection<O> objs, final E struct, final Collection<Field> fields, final Class<A> annoClazz) {
		Iterator<O> objIte = objs.iterator();
		for (Field field: fields) {
			if (annoClazz != null && field.getAnnotation(annoClazz) == null) continue;
			Object obj = null;
			try {
				obj = objIte.next();
			} catch (NoSuchElementException e) {
				throw new RuntimeException("row not found. class[" + struct.getClass().getName() + "] field[" + field.getName() + "]", e);
			}
			settingObjectToStruct(obj, struct, field);
		}
	}

	/**
	 * Mapから構造体を生成する。<br/>
	 *
	 * @param clazz 構造体型
	 * @param map Map
	 * @return 構造体
	 */
	public static <E> E createStruct(final Class<E> clazz, final Map<String, Object> map) {
		return StructUtil.createStruct(clazz, map, null);
	}

	/**
	 * Mapから構造体を生成する。<br/>
	 *
	 * @param clazz 構造体型
	 * @param map Map
	 * @param annoClazz 処理対象アノテーション型
	 * @return 構造体
	 */
	@SuppressWarnings("unchecked")
	public static <E, A extends Annotation> E createStruct(final Class<E> clazz, final Map<String, Object> map, final Class<A> annoClazz) {
		ClassInfo structClazzInfo = ClassInfoManager.getInstance().getClassInfo(clazz);
		E struct = (E)ReflectUtil.newInstance(structClazzInfo.getType());
		clearStruct(struct, annoClazz);
		settingMapToStruct(map, struct, annoClazz);
		return struct;
	}

	/**
	 * Mapを構造体に設定する。
	 *
	 * @param map Map
	 * @param struct 構造体
	 */
	public static <E> void settingMapToStruct(final Map<String, Object> map, final E struct) {
		settingMapToStruct(map, struct, null);
	}

	/**
	 * Mapを構造体に設定する。
	 *
	 * @param map Map
	 * @param struct 構造体
	 * @param annoClazz 処理対象アノテーション型
	 */
	public static <E, A extends Annotation> void settingMapToStruct(final Map<String, Object> map, final E struct, final Class<A> annoClazz) {
		settingMapToStruct(map, struct, ReflectUtil.getFields(struct.getClass()), annoClazz);
	}

	/**
	 * Mapを構造体に設定する。
	 *
	 * @param map Map
	 * @param struct 構造体
	 * @param fields Fieldコレクション
	 * @param annoClazz 処理対象アノテーション型
	 */
	protected static <E, A extends Annotation>
			void settingMapToStruct(final Map<String, Object> map, final E struct, final Collection<Field> fields, final Class<A> annoClazz) {
		for (String key: map.keySet()) {
			Field field = ReflectUtil.getField(struct, key);
			if (field == null) {
				String fieldName = NameUtil.addPrefix(key, 'm');
				field = ReflectUtil.getField(struct, fieldName);
			}
			if (field == null) {
				String fieldName = NameUtil.addPrefix(key, '_');
				field = ReflectUtil.getField(struct, fieldName);
			}
			if (field == null) continue;
			if (annoClazz != null && field.getAnnotation(annoClazz) == null) continue;

			Object value = map.get(key);
			settingObjectToStruct(value, struct, field);
		}
	}

	/**
	 * オブジェクトを構造体に設定する。
	 *
	 * @param obj オブジェクト
	 * @param struct 構造体
	 * @param field Field
	 * @return 設定できた場合true。
	 */
	protected static <E> boolean settingObjectToStruct(final Object obj, final E entity, final Field field) {
		Type genType = field.getGenericType();
		Object convertedObject = null;
		Class<?> fieldClazz = null;
		if (genType instanceof TypeVariable) {
			ClassInfo classInfo = ClassInfoManager.getInstance().getClassInfo(entity.getClass());
			fieldClazz = classInfo.getGenericsClass(field.getName());
		} else {
			fieldClazz = field.getType();
		}
		convertedObject = TypeConvertUtil.convert(fieldClazz, obj);
		if (convertedObject == null) return false;

		return ReflectUtil.setFieldValue(entity, field, convertedObject);
	}

	/**
	 * 構造体をクリアする。
	 *
	 * @param struct 構造体
	 * @param fields Fieldコレクション
	 * @param annoClazz 処理対象アノテーション型
	 */
	public static <E, A extends Annotation>
			void clearStruct(final E struct, final Class<A> annoClazz) {
		clearStruct(struct, ReflectUtil.getFields(struct.getClass()), annoClazz);
	}

	/**
	 * 構造体をクリアする。
	 *
	 * @param struct 構造体
	 * @param fields Fieldコレクション
	 * @param annoClazz 処理対象アノテーション型
	 */
	public static <E, A extends Annotation>
			void clearStruct(final E struct, final Collection<Field> fields, final Class<A> annoClazz) {
		for (Field field: fields) {
			if (annoClazz != null && field.getAnnotation(annoClazz) == null) continue;

			ReflectUtil.setFieldValue(struct, field, null);
		}
	}



} // end-class
