package option.gad.core.dao;

import java.util.StringTokenizer;
import java.util.Vector;


/**
 * 保存データの 1 つのセクションを表す。
 * 
 * @since		2005/01/07
 * @author		fujita
 * @version	2005/01/07
 * 
 * Copyright (c) 2004 KITec Inc,.. All rights reserved.
 */
public class Section {
	/** セクションの開始を表す文字列 */
	public static final String SECTION_START = "[";
	/** セクションの終了を表す文字列 */
	public static final String SECTION_END   = "]";

	/** ファイルの終端を表す定数 */	
	public static final int TYPE_EOF     = 0;
	/** 空白を表す定数 */
	public static final int TYPE_SPACE   = 1;
	/** セクションを表す定数 */
	public static final int TYPE_SECTION = 2;
	/** データを表す定数 */	
	public static final int TYPE_DATA    = 3;

	/**
	 * 指定された文字列を、セクションを表す文字列へ変換する。
	 * 
	 * @param		sectionName		文字列
	 * @return	セクションを表す文字列
	 * @since		2005/01/07
	 * @author		fujita
	 * @version	2005/01/07
	 */
	public static String toSectionString(final String sectionName) {
		return SECTION_START + sectionName + SECTION_END;		
	}

	/**
	 * 指定された文字列を種別を返す。
	 * 
	 * @param		str		判定する文字列
	 * @return	指定された文字列を種別
	 * @see		#TYPE_EOF
	 * @see		#TYPE_SPACE
	 * @see		#TYPE_SECTION
	 * @see		#TYPE_DATA
	 * @since		2005/01/07
	 * @author		fujita
	 * @version	2005/01/07
	 */
	public static int getType(final String str) {
		if (str == null) {
			return TYPE_EOF;
		}

		str.trim();
		if (str.length() == 0) {
			return TYPE_SPACE;
		}
		if (str.startsWith(SECTION_START) && str.endsWith(SECTION_END)) {
			return TYPE_SECTION;
		}
		return TYPE_DATA;
	}


	/** セクション名 */
	private String mSectionName;
	/** セクションに含まれるデータ群 */
	private Vector mData;


	/**
	 * 空の内容を持つ新たな Section を作成する。
	 * 
	 * @since		2005/01/07
	 * @author		fujita
	 * @version	2005/01/07
	 */
	public Section() {
		mData = new Vector();
	}

	/**
	 * 指定された文字列を、セクション名として設定する。
	 * 
	 * @param		section		文字列
	 * @since		2005/01/07
	 * @author		fujita
	 * @version	2005/01/07
	 */
	public void setSectionName(final String section) {
		mSectionName = section;
	}

	/**
	 * セクション名を返す。
	 * 
	 * @return	セクション名
	 * @since		2005/01/07
	 * @author		fujita
	 * @version	2005/01/07
	 */
	public String getSectionName() {
		return mSectionName;
	}

	/**
	 * 指定された data を、このセクションの
	 * データとして追加する。
	 * 
	 * @param		data	データ
	 * @since		2005/01/07
	 * @author		fujita
	 * @version	2005/01/07
	 */
	public void addData(final String data) {
		mData.addElement(data);
	}

	/**
	 * このセクションに含まれるデータの数を返す。
	 * 
	 * @return	セクションのデータ数
	 * @since		2005/01/07
	 * @author		fujita
	 * @version	2005/01/07
	 */
	public int getDataCount() {
		return mData.size();
	}

	/**
	 * 指定されたインデックスのデータを返す。
	 * 
	 * @param		index	データのインデックス
	 * @return	データ
	 * @since		2005/01/07
	 * @author		fujita
	 * @version	2005/01/07
	 */
	public String getData(final int index) {
		return (String)mData.elementAt(index);
	}

	/**
	 * 指定されたインデックスをデータを
	 * 指定された区切り文字で分割して返す。
	 * 
	 * @param		index		データのインデックス
	 * @param		separator	区切り文字
	 * @return	区切り文字で分割されたデータ群
	 * @since		2005/01/07
	 * @author		fujita
	 * @version	2005/01/07
	 */
	public String[] getSplitData(final int index, final String separator) {
		StringTokenizer st = new StringTokenizer(getData(index), separator);
		String[] tokens = new String[st.countTokens()];
		for (int i = 0; i < tokens.length; i++) {
			tokens[i] = st.nextToken(); 
		}
		return tokens;
	}

	/**
	 * 指定されたセクション名と、このセクションが
	 * 同じ名前を持つかどうかを判定する。
	 * 
	 * @param		sectionName		セクション名
	 * @return	指定されたセクション名と、このセクションの
	 * 				名前が同じ場合、true。そうでない場合は、false
	 * @since		2005/01/07
	 * @author		fujita
	 * @version	2005/01/07
	 */
	public boolean equals(final String sectionName) {
		return mSectionName.equals(sectionName);
	}

	/**
	 * @see		java.lang.Object#toString()
	 * @since		2005/01/07
	 * @author		fujita
	 * @version	2005/01/07
	 */
	public String toString() {
		StringBuffer sb = new StringBuffer("[");
		sb.append(mSectionName);
		sb.append("]");
		final int count = getDataCount();
		for (int i = 0; i < count; i++) {
			sb.append(getData(i));
			sb.append("\n");
		}
		return sb.toString();
	}
}
