/*
 * Copyright (c) 2010-2012 KITec Inc,.. All rights reserved.
 */
package option.gad.core.dao;

import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.util.Collection;
import java.util.LinkedList;

import jp.kitec.lib.annotation.Reflectable;
import jp.kitec.lib.dao.ObjectFolderDao;
import jp.kitec.lib.util.ReflectUtil;
import jp.kitec.lib.util.tree.ObjectFolder;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;



/**
 * Input/OutputStream版、SaveFileDao
 *
 * @author $Author$
 * @version $Revision$ $Date::                           $
 */
@Reflectable
public abstract class GdAbstractIOStreamSaveFileDao<DOC>
	extends GdAbstractSaveFileDao<InputStream, OutputStream, DOC> {

	//------------------------------------------------------------------
	//- fields
	//------------------------------------------------------------------

	/** ログ */
	private static final Log mLog = LogFactory.getLog(GdAbstractIOStreamSaveFileDao.class);



	//------------------------------------------------------------------
	//- constructors
	//------------------------------------------------------------------

	/**
	 * コンストラクタ
	 */
	protected GdAbstractIOStreamSaveFileDao() {
	}



	//------------------------------------------------------------------
	//- methods
	//------------------------------------------------------------------

	protected String configHeaderEncoding() {
		return "Shift_JIS";
	}

	protected String configInputEncoding() {
		return "MS932";
	}

	protected String configOutputEncoding() {
		return "MS932";
	}

	protected String configFolderName() {
		return "data";
	}

	protected String configRowSuffix() {
		return "Row";
	}

	protected String createRowName(String fieldName) {
		return fieldName + configRowSuffix();
	}

	/**
	 * ドキュメントデータを永続層に保存する。
	 *
	 * @param os {@link OutputStream}
	 * @param doc ドキュメント
	 */
	public void save(OutputStream os, DOC doc) {
		save(os, doc, configHeaderEncoding(), configOutputEncoding());
	}

	/**
	 * ドキュメントデータを永続層に保存する。
	 *
	 * @param os {@link OutputStream}
	 * @param doc ドキュメント
	 * @param encoding 出力文字エンコーディング
	 */
	public void save(OutputStream os, DOC doc, String encoding) {
		save(os, doc, encoding, encoding);
	}

	/**
	 * ドキュメントデータを永続層に保存する。
	 *
	 * @param os {@link OutputStream}
	 * @param doc ドキュメント
	 * @param headerEncoding ヘッダ文字エンコーディング
	 * @param outputEncoding 出力文字エンコーディング
	 */
	public void save(OutputStream os, DOC doc, String headerEncoding, String outputEncoding) {
		ObjectFolder rootFolder = new ObjectFolder(configFolderName());
		for (Field field: ReflectUtil.getFields(doc)) {
			Reflectable annoReflectable = field.getAnnotation(Reflectable.class);
			if (annoReflectable == null) {
				mLog.debug("Reflectable not found from " + field.getType() + ".");
				continue;
			}
			String fieldName = annoReflectable.name();

			Object fieldValue = ReflectUtil.getFieldValue(doc, fieldName);
			if (Collection.class.isAssignableFrom(field.getType())) {
				String rowName = createRowName(fieldName);
				saveFolders(rootFolder, fieldName, rowName, (Collection<?>)fieldValue);
			} else {
				saveFolder(rootFolder, fieldName, fieldValue);
			}
		}

		ObjectFolderDao dao = new ObjectFolderDao();
		dao.setOutput(os);
		dao.save(rootFolder, headerEncoding, outputEncoding);
	}

	/**
	 * 永続層からドキュメントデータを読み込む。
	 *
	 * @param is {@link InputStream}
	 * @param doc ドキュメント
	 */
	public void load(InputStream is, DOC doc) {
		load(is, doc, configInputEncoding());
	}

	/**
	 * 永続層からドキュメントデータを読み込む。
	 *
	 * @param is {@link InputStream}
	 * @param doc ドキュメント
	 * @param encoding 入力文字エンコーディング
	 */
	public void load(InputStream is, DOC doc, String encoding) {
		ObjectFolderDao dao = new ObjectFolderDao();
		dao.setInput(is);
		ObjectFolder rootFolder = dao.load(encoding);

		for (Field field: ReflectUtil.getFields(doc)) {
			Reflectable annoReflectable = field.getAnnotation(Reflectable.class);
			if (annoReflectable == null) {
				mLog.debug("Reflectable not found from " + field.getType() + ".");
				continue;
			}
			String fieldName = annoReflectable.name();

			if (Collection.class.isAssignableFrom(field.getType())) {
				String rowName = createRowName(fieldName);
				Collection<?> fieldValue = loadFolders(rootFolder, fieldName, rowName);
				if (fieldValue == null) fieldValue = new LinkedList<Object>();
				ReflectUtil.setFieldValue(doc, fieldName, fieldValue);
			} else {
				Object fieldValue = loadFolder(rootFolder, fieldName);
				if (fieldValue == null) continue;
				ReflectUtil.setFieldValue(doc, fieldName, fieldValue);
			}
		}
	}



} // end-class
