/*
 * Copyright (c) 2010-2014 KITec Inc,.. All rights reserved.
 */
package option.gad.core.dao;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import jp.kitec.lib.annotation.Reflectable;
import jp.kitec.lib.util.ReflectUtil;
import jp.kitec.lib.util.StringUtil;
import jp.kitec.lib.util.tree.ObjectFolder;
import option.gad.core.annotation.GdPrefix;
import option.gad.core.annotation.GdProperty;
import option.gad.core.annotation.GdVersion;
import option.gad.core.util.ObjectFolderUtil;
import option.gad.core.util.ObjectNodeUtil;

import org.apache.commons.lang.math.NumberUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;



/**
 * 保存ファイルDaoの抽象実装。
 *
 * @author $Author$
 * @version $Revision$ $Date::                           $
 */
public abstract class GdAbstractSaveFileDao<IN, OUT, DOC> implements GdSaveFileDao<IN, OUT, DOC> {

	//------------------------------------------------------------------
	//- enum
	//------------------------------------------------------------------

	/** 保存ファイル状態 */
	public enum SaveFileStatus {
		NORMAL
		,WARN
		,ERROR
	}



	//------------------------------------------------------------------
	//- fields
	//------------------------------------------------------------------

	/** ログ */
	private Log mLog = LogFactory.getLog(GdAbstractSaveFileDao.class);

	/** 出力メッセージリストMap */
	protected HashMap<SaveFileStatus, List<String>> mMessageListMap =
		new HashMap<SaveFileStatus, List<String>>();

	/** 保存ファイル状態 */
	protected SaveFileStatus mStatus = SaveFileStatus.NORMAL;



	//------------------------------------------------------------------
	//- constructors
	//------------------------------------------------------------------

	/**
	 * コンストラクタ
	 */
	protected GdAbstractSaveFileDao() {
		init();
	}



	//------------------------------------------------------------------
	//- methods
	//------------------------------------------------------------------

	protected void init() {
		for (SaveFileStatus status: SaveFileStatus.values()) {
			mMessageListMap.put(status, new ArrayList<String>());
		}
	}

	protected <T> void saveFolder(ObjectFolder rootFolder, String targetName, T target) {
		if (target == null) return;
		rootFolder.addChild(ObjectFolderUtil.createFolder(target, targetName, GdProperty.class));
	}

	protected <T> void saveFolders(ObjectFolder rootFolder, String targetName, String targetRowName, Collection<T> targets) {
		if (targets == null || targets.isEmpty()) return;
		ObjectFolder folder = ObjectFolderUtil.createFolder(targets, targetName, GdProperty.class);
		for (T target: targets) {
			folder.addChild(ObjectFolderUtil.createFolder(target, targetRowName, GdProperty.class), true);
		}
		rootFolder.addChild(folder);
	}

	protected abstract Class<?>[] configLoadClasses();

	protected Class<?> findClass(String clazzName, int ver) {
		String clazzVerName = null;
		if (ver == 0) {
			clazzVerName = clazzName;
		} else {
			String version = "V" + ver;
			clazzVerName = clazzName + version;
		}

		Class<?> result = null;
		for (Class<?> clazz: configLoadClasses()) {
			Reflectable annoReflectable = clazz.getAnnotation(Reflectable.class);
			assert annoReflectable != null: "Reflectable not found from " + clazz.getSimpleName()  + ".";
			GdPrefix annoPrefix = clazz.getAnnotation(GdPrefix.class);
			String configClazzName = annoReflectable.name();
			String clazzPrefixName = clazzVerName;
			if (annoPrefix != null) clazzPrefixName = annoPrefix.name() + clazzVerName;

			if (clazzName.equals(configClazzName)) {
				result = clazz;
				break;
			} else if (clazzVerName.equals(configClazzName)) {
				result = clazz;
				break;
			} else if (clazzPrefixName.equals(configClazzName)) {
				result = clazz;
				break;
			}
		}

		assert !StringUtil.isEmpty(clazzName): "clazzName is empty.";
		if (result == null) {
			result = ReflectUtil.getClass(clazzName);
		}
		return result;
	}

	protected <T> T loadFolder(ObjectFolder rootFolder, String targetName) {
		return loadFolder(rootFolder, targetName, "");
	}

	protected <T> T loadFolder(ObjectFolder rootFolder, String targetName, Class<?> defaultClazz) {
		Reflectable annoReflectable = defaultClazz.getAnnotation(Reflectable.class);
		assert annoReflectable != null: "Reflectable not found from " + defaultClazz.getSimpleName()  + ".";
		return loadFolder(rootFolder, targetName, annoReflectable.name());
	}

	protected <T> T loadFolder(ObjectFolder rootFolder, String targetName, String defaultType) {
		Collection<ObjectFolder> folders = ObjectFolderUtil.findChildFolders(rootFolder, targetName);
		if (folders.size() != 1) return null;

		ObjectFolder folder = folders.iterator().next();
		return createObject(folder, targetName, defaultType);
	}

	protected <T> Collection<T> loadFolders(ObjectFolder rootFolder, String targetName, String targetRowName) {
		return loadFolders(rootFolder, targetName, targetRowName, "");
	}

	protected <T> Collection<T> loadFolders(ObjectFolder rootFolder, String targetName, String targetRowName, Class<?> defaultClazz) {
		Reflectable annoReflectable = defaultClazz.getAnnotation(Reflectable.class);
		assert annoReflectable != null: "Reflectable not found from " + defaultClazz.getSimpleName()  + ".";
		return loadFolders(rootFolder, targetName, targetRowName, annoReflectable.name());
	}

	protected <T> Collection<T> loadFolders(ObjectFolder rootFolder, String targetName, String targetRowName, String defaultType) {
		Collection<ObjectFolder> targetFolders = ObjectFolderUtil.findChildFolders(rootFolder, targetName);
		if (targetFolders.size() != 1) return null;
		ObjectFolder targetFolder = targetFolders.iterator().next();

		String type = ObjectNodeUtil.toString(targetFolder.getNode("type"));
		if (StringUtil.isEmpty(type)) type = ArrayList.class.getName();
		Collection<T> targetCollection = ReflectUtil.newInstance(type);

		Collection<ObjectFolder> childFolderList = ObjectFolderUtil.findChildFolders(targetFolder, targetRowName);
		for (ObjectFolder childFolder: childFolderList) {
			T targetRow = createObject(childFolder, targetName, defaultType);
			if (targetRow == null) continue;

			targetCollection.add(targetRow);
		}
		return targetCollection;
	}

	@SuppressWarnings("unchecked")
	protected <T> T createObject(ObjectFolder folder, String targetName, String defaultType) {
		String type = ObjectNodeUtil.toString(folder.getNode("type"));
		if (StringUtil.isEmpty(type)) type = defaultType;
		if (StringUtil.isEmpty(type)) {
			mLog.info("type not found from '" + targetName + "' node.");
			return null;
		}

		Class<?> clazz = null;
		for (int ver = getDaoVersion(); ver >= 0; ver--) {
			try {
				clazz = findClass(type, ver);
				if (clazz != null) break;
			} catch (Exception e) {
			}
		}
		if (clazz == null) {
			if (StringUtil.isEmpty(defaultType)) {
				mLog.info("default-type not found.");
				return null;
			}
			clazz = findClass(defaultType, 0);
			if (clazz == null) {
				mLog.info("class not found from '" + type + "' type.");
				return null;
			}
		}

		return (T)ObjectFolderUtil.createObject(clazz, folder, GdProperty.class);
	}

	/**
	 * Daoバージョンを取得する。
	 *
	 * @return Daoバージョン
	 */
	protected int getDaoVersion() {
		GdVersion annoVersion = this.getClass().getAnnotation(GdVersion.class);
		String strVer = null;
		if (annoVersion != null) {
			strVer = annoVersion.value().replaceAll("\\.", "");
		} else {
			strVer = "";
		}
		return NumberUtils.toInt(strVer, 0);
	}

	/**
	 * バージョン番号を取得する。<br/>
	 * デフォルトでは、Daoにアノテーションで定義してあるバージョン番号が返される。<br/>
	 *
	 * @return バージョン番号
	 */
	public String getSaveDataVersion() {
		GdVersion annoVersion = this.getClass().getAnnotation(GdVersion.class);
		return StringUtil.emptyToDefault(annoVersion.value(), "Unknown");
	}

	/**
	 * ビルド番号を取得する。
	 *
	 * @return ビルド番号
	 */
	protected Integer getBuildNumber() {
		String buildNumber = "0";
		try {
			Properties properties = new Properties();
			properties.load(this.getClass().getClassLoader().getResourceAsStream("buildnumber.properties"));
			buildNumber = (String)properties.get("build.number");
		} catch (Exception e) {
		}
		return Integer.parseInt(buildNumber);
	}

	/**
	 * 出力メッセージリストを取得する。
	 *
	 * @return 出力メッセージリスト
	 */
	@Deprecated
	public List<String> getMessageList() {
		return mMessageListMap.get(SaveFileStatus.NORMAL);
	}

	/**
	 * 全レベルの出力メッセージリストを取得する。
	 *
	 * @return 全レベルの出力メッセージリスト
	 */
	public List<String> getAllMessageList() {
		ArrayList<String> allMessageList = new ArrayList<String>();
		for (SaveFileStatus status: SaveFileStatus.values()) {
			allMessageList.addAll(mMessageListMap.get(status));
		}
		return allMessageList;
	}

	/**
	 * 出力メッセージリストMapを取得する。
	 *
	 * @return 出力メッセージリストMap
	 */
	public Map<SaveFileStatus, List<String>> getMessageListMap() {
		return mMessageListMap;
	}

	/**
	 * 全出力メッセージをクリアする。
	 */
	public void clearAllMessages() {
		for (SaveFileStatus status: SaveFileStatus.values()) {
			mMessageListMap.get(status).clear();
		}
	}

	/**
	 * 保存ファイル状態を設定する。
	 *
	 * @param status 保存ファイル状態
	 */
	public void setStatus(SaveFileStatus status) {
		mStatus = status;
	}

	/**
	 * 保存ファイル状態を更新する。<br/>
	 * 状態がより深刻な場合のみ更新する。<br/>
	 *
	 * @param status 保存ファイル状態
	 */
	public void updateStatus(SaveFileStatus status) {
		if (mStatus.ordinal() < status.ordinal()) mStatus = status;
	}

	/**
	 * 保存ファイル状態を取得する。
	 *
	 * @return 保存ファイル状態
	 */
	public SaveFileStatus getStatus() {
		return mStatus;
	}



} // end-class
