/*
 * Copyright (c) 2009-2013 KITec Inc,.. All rights reserved.
 */
package option.gad.core.util;

import java.awt.Color;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import jp.kitec.lib.annotation.Reflectable;
import jp.kitec.lib.util.ClassInfo;
import jp.kitec.lib.util.ClassInfoManager;
import jp.kitec.lib.util.NameUtil;
import jp.kitec.lib.util.ObjectUtil;
import jp.kitec.lib.util.ReflectUtil;
import jp.kitec.lib.util.StringUtil;
import jp.kitec.lib.util.tree.ObjectFolder;
import jp.kitec.lib.util.tree.ObjectNode;
import option.gad.core.annotation.GdTransient;
import option.gad.core.dxo.GdTypeConvertable;
import option.gad.core.dxo.TypeConvertUtil;



/**
 * ObjectFolderユーティリティクラス
 *
 * @author $Author$
 * @version $Revision$ $Date::                           $
 */
public class ObjectFolderUtil extends jp.kitec.lib.util.ObjectFolderUtil {

	//------------------------------------------------------------------
	//- constructors
	//------------------------------------------------------------------

	/**
	 * コンストラクタ
	 */
	protected ObjectFolderUtil() {
	}



	//------------------------------------------------------------------
	//- methods
	//------------------------------------------------------------------

	/**
	 * オブジェクトからObjectFolderを生成する。<br/>
	 * ObjectFolder名は、クラス名を使用する。<br/>
	 *
	 * @param obj ObjectFolder化するオブジェクト
	 * @return ObjectFolder
	 */
	public static <A extends Annotation> ObjectFolder createFolder(Object obj) {
		return createFolder(obj, (Class<A>)null);
	}

	/**
	 * オブジェクトからObjectFolderを生成する。<br/>
	 * ObjectFolder名は、クラス名を使用する。<br/>
	 *
	 * @param obj ObjectFolder化するオブジェクト
	 * @param annoClazz 処理対象アノテーション型
	 * @return ObjectFolder
	 */
	public static <A extends Annotation> ObjectFolder createFolder(Object obj, Class<A> annoClazz) {
		String name = NameUtil.correctCamelCase(obj.getClass().getSimpleName());
		return createFolder(obj, name, annoClazz);
	}

	/**
	 * オブジェクトからObjectFolderを生成する。
	 *
	 * @param obj ObjectFolder化するオブジェクト
	 * @param name ObjectFolder名
	 * @return ObjectFolder
	 */
	public static <A extends Annotation> ObjectFolder createFolder(Object obj, String name) {
		return createFolder(obj, name, null);
	}

	/**
	 * オブジェクトからObjectFolderを生成する。
	 *
	 * @param obj ObjectFolder化するオブジェクト
	 * @param name ObjectFolder名
	 * @param annoClazz 処理対象アノテーション型
	 * @return ObjectFolder
	 */
	public static <A extends Annotation> ObjectFolder createFolder(Object obj, String name, Class<A> annoClazz) {
		ObjectFolder currerntFolder = new ObjectFolder(name);
		addObject(currerntFolder, name, obj, annoClazz);
		return currerntFolder;
	}

	/**
	 * ObjectFolderにオブジェクトを追加する。
	 *
	 * @param folder ObjectFolder
	 * @param obj 追加するオブジェクト
	 * @param annoClazz 処理対象アノテーション型
	 */
	public static <A extends Annotation> void addObject(ObjectFolder folder, Object value, Class<A> annoClazz) {
		addObject(folder, null, value, annoClazz);
	}

	/**
	 * ObjectFolderにオブジェクトを追加する。
	 *
	 * @param folder ObjectFolder
	 * @param name オブジェクト名
	 * @param obj 追加するオブジェクト
	 * @param annoClazz 処理対象アノテーション型
	 */
	public static <A extends Annotation> void addObject(ObjectFolder folder, String name, Object value, Class<A> annoClazz) {
		if (value == null) return;

		ClassInfo valueClazzInfo = ClassInfoManager.getInstance().getClassInfo(value.getClass());

		if (isStdType(value)) {
			addPrimitive(folder, value);
			return;
		}

		if (valueClazzInfo.getType().isArray()) {
			addArray(folder, name, (Object[])value, annoClazz);
			return ;
		}

		addStruct(folder, value, annoClazz);
	}

	/**
	 * ObjectFolderに構造体を追加する。
	 *
	 * @param folder ObjectFolder
	 * @param obj 追加するオブジェクト
	 * @param annoClazz 処理対象アノテーション型
	 */
	public static <A extends Annotation> void addStruct(ObjectFolder folder, Object obj, Class<A> annoClazz) {
		if (obj == null) return;

		String clazzName = createTypeName(obj);
		folder.addChild(new ObjectNode("type", clazzName));

		for (Field field: ReflectUtil.getFields(obj)) {
			if (field.getAnnotation(GdTransient.class) != null) continue;
			if (annoClazz != null && field.getAnnotation(annoClazz) == null) continue;

			Object value = ReflectUtil.getFieldValue(obj, field);
			if (value == null) continue;

			String name = NameUtil.removePrefix(field.getName());
			if (isStdType(value)) {
				addValue(folder, name, value);
			} else if (value instanceof GdTypeConvertable) {
				addValue(folder, name, value);
			} else {
				ObjectFolder childFolder = createFolder(value, name, annoClazz);
				folder.addChild(childFolder);
			}
		}
	}

	/**
	 * ObjectFolderに配列を追加する。
	 *
	 * @param folder ObjectFolder
	 * @param name 配列名
	 * @param obj 追加する配列
	 */
	protected static <A extends Annotation> void addArray(ObjectFolder folder, String name, Object[] objs, Class<A> annoClazz) {
		if (objs == null) return;

		name = NameUtil.removePrefix(name);
		name = NameUtil.correctCamelCase(name);

		for (Object obj: objs) {
			ObjectFolder arrayFolder = new ObjectFolder(name);
			addObject(arrayFolder, name, obj, annoClazz);
			folder.addChild(arrayFolder, true);
		}
	}

	/**
	 * ObjectFolderにPrimitiveを追加する。
	 *
	 * @param folder ObjectFolder
	 * @param value 値
	 */
	protected static void addPrimitive(ObjectFolder folder, Object value) {
		String clazzName = createTypeName(value);
		addValue(folder, "type", clazzName);
		addValue(folder, "value", value);
	}

	/**
	 * ObjectFolderに値を追加する。
	 *
	 * @param folder ObjectFolder
	 * @param value 値
	 */
	protected static void addValue(ObjectFolder folder, Object value) {
		addValue(folder, "value", value);
	}

	/**
	 * ObjectFolderに値を追加する。
	 *
	 * @param folder ObjectFolder
	 * @param name 値名
	 * @param value 値
	 */
	protected static void addValue(ObjectFolder folder, String name, Object value) {
		if (name == null) return;
		if (value == null) return;

		name = NameUtil.removePrefix(name);
		name = NameUtil.correctCamelCase(name);
		String strValue = TypeConvertUtil.convertString(value);
		if (StringUtil.isEmpty(strValue)) return;

		folder.addChild(new ObjectNode(name, strValue));
	}

	/**
	 * ObjectFolderからオブジェクトを生成する。
	 *
	 * @param clazz オブジェクト型
	 * @param folder ObjectFolder
	 * @param annoClazz 処理対象アノテーション型
	 * @return 構造体
	 */
	public static <T, A extends Annotation> T createObject(Class<T> clazz, ObjectFolder folder, Class<A> annoClazz) {
		if (isStdType(clazz)) {
			return createPrimitive(clazz, folder);
		}

//FIXME:配列処理は一時凍結。基本的には無くす方向。
//		Collection<ObjectFolder> childFolders = collectChildFolders(folder, 1);
//		if (childFolders.size() > 1) {
//			Set<String> nameSet = new HashSet<String>();
//			for (ObjectFolder childFolder: childFolders) {
//				nameSet.add(childFolder.getName());
//			}
//			if (nameSet.size() == 1) return (T)createArray(folder);
//		}

		return createStruct(clazz, folder, annoClazz);
	}

	/**
	 * ObjectFolderから構造体を生成する。
	 *
	 * @param clazz 構造体型
	 * @param folder ObjectFolder
	 * @param annoClazz 処理対象アノテーション型
	 * @return 構造体
	 */
	protected static <T, A extends Annotation> T createStruct(Class<T> clazz, ObjectFolder folder, Class<A> annoClazz) {
		List<ObjectNode> nodeList = folder.getChildren();

		Map<String, Object> map = new LinkedHashMap<String, Object>();
		for (ObjectNode node: nodeList) {
			map.put(node.getName(), node.getObject());
		}

		return StructUtil.createStruct(clazz, map, annoClazz);
	}

	/**
	 * 配列を生成する。
	 *
	 * @param folder ObjectFoloder
	 * @return 配列
	 */
	protected static Object[] createArray(ObjectFolder folder) {
		List<ObjectNode> nodeList = folder.getChildren();

		List<Object> objList = new ArrayList<Object>();
		for (ObjectNode nodeFolder: nodeList) {
			objList.add(((ObjectFolder)nodeFolder).getNode("value").getObject());
		}
		return objList.toArray(new Object[objList.size()]);
	}

	/**
	 * Primitiveを生成する。
	 *
	 * @param folder ObjectFoloder
	 * @return Primitive
	 */
	public static <T> T createPrimitive(Class<T> clazz, ObjectFolder folder) {
		ObjectNode valueNode = folder.getNode("value");
		String value = ObjectFolder.getNodeString(valueNode);
		return TypeConvertUtil.convertObject(clazz, value);
	}

	/**
	 * 保存時の種別名を生成する。
	 *
	 * @param obj 対象オブジェクト
	 * @return 保存時の種別名
	 */
	protected static String createTypeName(Object obj) {
		if (obj == null) return null;

		String clazzName = null;
		Class<?> objClazz = obj.getClass();
		Reflectable annoReflectable = objClazz.getAnnotation(Reflectable.class);
		if (annoReflectable != null) clazzName = annoReflectable.name();
		if (StringUtil.isEmpty(clazzName)) clazzName = objClazz.getName();

		return clazzName;
	}

	/**
	 * JRE標準クラスかをチェックする。
	 *
	 * @param obj 対象オブジェクト
	 * @return JRE標準クラスの場合true
	 */
	public static boolean isStdType(Object obj) {
		return isStdType(obj.getClass());
	}

	/**
	 * JRE標準クラスかをチェックする。
	 *
	 * @param clazz 対象Class
	 * @return JRE標準クラスの場合true
	 */
	private static boolean isStdType(Class<?> clazz) {
		return ObjectUtil.isLangType(clazz)
			|| Color.class.isAssignableFrom(clazz)
			|| Timestamp.class.isAssignableFrom(clazz);
	}



} // end-class
