/*
 * Copyright (c) 2009-2014 KITec Inc,.. All rights reserved.
 */
package option.gad.core.dao;

import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.kitec.lib.io.AbstFile;
import jp.kitec.lib.util.ClassInfo;
import jp.kitec.lib.util.ClassInfoManager;
import jp.kitec.lib.util.ListNotNullDecorator;
import jp.kitec.lib.util.NameUtil;
import jp.kitec.lib.util.StringUtil;
import option.gad.core.annotation.GdBitField;
import option.gad.core.annotation.GdByteLength;
import option.gad.core.annotation.GdProperty;
import option.gad.core.annotation.GdQuote;
import option.gad.core.db.NoUniqueRuntimeException;
import option.gad.core.entity.GdEntity;
import option.gad.core.inject.GdInject;
import option.gad.core.io.FileIOUtil;
import option.gad.core.util.StructUtil;
import option.gad.core.util.csv.Csv;
import option.gad.core.util.csv.CsvBuilder;
import option.gad.core.util.csv.CsvUtil;
import option.gad.core.util.csv.CsvUtil.BitFieldComparator;
import option.gad.core.util.csv.GdCsvColumn;

import org.apache.commons.lang.math.NumberUtils;



/**
 * CSVファイルへアクセスするDaoの抽象実装
 *
 * @author $Author$
 * @version $Revision$ $Date::                           $
 */
public abstract class AbstractCsvDao<E> {

	//------------------------------------------------------------------
	//- delegation components
	//------------------------------------------------------------------

	@GdInject
	protected CsvBuilder mCsvBuilder = null;



	//------------------------------------------------------------------
	//- fields
	//------------------------------------------------------------------

	/** リソース名 */
	protected String mResourceName = null;

	/** ヘッダ状態。ある場合true */
	protected boolean mHeadExist = true;

	/** Entityクラス情報 */
	protected ClassInfo mEntityClazzInfo = null;

	/** Csv結果セット */
	protected Csv mCsv = null;

	/** Csv比較処理Map */
	protected Map<String, Comparator<String>> mCompMap = null;

	protected AbstFile mAbstFile = null;

	protected File mFile = null;

	protected OutputStream mOutputStream = null;



	//------------------------------------------------------------------
	//- constructors
	//------------------------------------------------------------------

	/**
	 * コンストラクタ
	 */
	protected AbstractCsvDao() {
		super();
		this.init();
	}



	//------------------------------------------------------------------
	//- initialize methods
	//------------------------------------------------------------------

	/**
	 * 初期化する。
	 */
	protected void init() {
		ClassInfo classInfo = ClassInfoManager.getInstance().getClassInfo(this.getClass());

		GdDao annoDao = classInfo.getType().getAnnotation(GdDao.class);
		if (annoDao == null) throw new IllegalStateException("GdDao is not defined in " + classInfo.getType().getName() + ".");
		this.mEntityClazzInfo = ClassInfoManager.getInstance().getClassInfo(annoDao.entity());

		GdEntity annoEntity = this.mEntityClazzInfo.getType().getAnnotation(GdEntity.class);
		if (annoEntity == null) throw new IllegalStateException("GdEntity is not defined in " + this.mEntityClazzInfo.getType().getName() + ".");
		this.mResourceName = annoEntity.name();
		this.mHeadExist = annoEntity.head();
		this.mCompMap = createCompMap();
	}

	/**
	 * デフォルトの文字エンコーディングを定義する。
	 */
	protected String configDefaultCharacterEncoding() {
		return "UTF-8";
	}

	/**
	 * カラム毎の比較処理を決定する
	 */
	protected Map<String, Comparator<String>> createCompMap() {
		HashMap<String, Comparator<String>> compMap = new HashMap<String, Comparator<String>>();

		Collection<Field> fields = mEntityClazzInfo.getFields();
		for (Field field: fields) {
			String fieldName = NameUtil.correctCamelCase(NameUtil.removePrefix(field.getName()));
			if (field.getAnnotation(GdBitField.class) != null) {
				compMap.put(fieldName, CsvUtil.comparatorMap.get(BitFieldComparator.class));
			}
			GdCsvColumn annoCsvColumn = field.getAnnotation(GdCsvColumn.class);
			if (annoCsvColumn != null) {
				compMap.put(fieldName, CsvUtil.comparatorMap.get(annoCsvColumn.comparator()));
			}
		}

		return compMap;
	}



	//------------------------------------------------------------------
	//- methods
	//------------------------------------------------------------------

	/**
	 * Entityをキーに、永続層からデータを取得する。<br/>
	 *
	 * @param entityKey 検索キー
	 * @param allows[0] trueの場合、値が存在しないカラムも検索に一致したと見なす。可変パラメータのため設定しなくても良い。
	 * @param allows[1] trueの場合、先頭に#が存在する行をコメントと見なし処理対象としない。可変パラメータのため設定しなくても良い。
	 * @return Entity
	 * @see #findOne(String[])
	 */
	public E findOne(final E entityKey, final boolean... allows) {
		return this.findOne(StructUtil.createStrings(entityKey), allows);
	}

	/**
	 * 文字列配列キーに、永続層からデータを取得する。<br/>
	 * データが複数取得できた場合は、IllegalStateExceptionをスローする。<br/>
	 * データが取得できなかった場合は、nullを返す。<br/>
	 *
	 * @param keys 検索キー
	 * @param allows[0] trueの場合、値が存在しないカラムも検索に一致したと見なす。可変パラメータのため設定しなくても良い。
	 * @param allows[1] trueの場合、先頭に#が存在する行をコメントと見なし処理対象としない。可変パラメータのため設定しなくても良い。
	 * @return Entity
	 * @throws IllegalStateException
	 */
	public E findOne(final String[] keys, final boolean... allows) {
		return findOne(Arrays.asList(keys), allows);
	}

	/**
	 * 文字列配列キーに、永続層からデータを取得する。<br/>
	 * データが複数取得できた場合は、IllegalStateExceptionをスローする。<br/>
	 * データが取得できなかった場合は、nullを返す。<br/>
	 *
	 * @param keys 検索キー
	 * @param allows[0] trueの場合、値が存在しないカラムも検索に一致したと見なす。可変パラメータのため設定しなくても良い。
	 * @param allows[1] trueの場合、先頭に#が存在する行をコメントと見なし処理対象としない。可変パラメータのため設定しなくても良い。
	 * @return Entity
	 * @throws IllegalStateException
	 */
	public E findOne(final Collection<String> keys, final boolean... allows) {
		if (mCsv == null) {
			loadResource();
		}
		if (mCsv == null) throw new IllegalArgumentException("master csv isn't load.");

		Csv dstCsv = CsvUtil.select(mCsv, keys, mCompMap, allows);
		if (dstCsv.mRowList.size() > 1) throw new NoUniqueRuntimeException(dstCsv.mRowList.size(), "keys[" + StringUtil.concat(keys, ",") + "]");
		if (dstCsv.mRowList.size() == 0) return null;

		return this.createEntity(dstCsv.mRowList.iterator().next());
	}

	/**
	 * 永続層から全データを取得する。
	 *
	 * @return Entityコレクション
	 * @see #findMany(String[])
	 */
	public Collection<E> findAll() {
		String[] keys = new String[this.mEntityClazzInfo.getFields().size()];
		for (@SuppressWarnings("unused")String key: keys) key = null;
		return this.findMany(keys);
	}

	/**
	 * Entityをキーに、永続層から複数のデータを取得する。
	 *
	 * @param entityKey 検索キー
	 * @param allows[0] trueの場合、値が存在しないカラムも検索に一致したと見なす。可変パラメータのため設定しなくても良い。
	 * @param allows[1] trueの場合、先頭に#が存在する行をコメントと見なし処理対象としない。可変パラメータのため設定しなくても良い。
	 * @return Entityコレクション
	 * @see #findMany(String[])
	 */
	public Collection<E> findMany(final E entityKey, final boolean... allows) {
		return this.findMany(StructUtil.createStrings(entityKey), allows);
	}

	/**
	 * 文字列配列をキーに、永続層から複数のデータを取得する。
	 *
	 * @param keys 検索キー
	 * @param allows[0] trueの場合、値が存在しないカラムも検索に一致したと見なす。可変パラメータのため設定しなくても良い。
	 * @param allows[1] trueの場合、先頭に#が存在する行をコメントと見なし処理対象としない。可変パラメータのため設定しなくても良い。
	 * @return Entityコレクション
	 */
	public Collection<E> findMany(final String[] keys, final boolean... allows) {
		return findMany(Arrays.asList(keys), allows);
	}

	/**
	 * 文字列配列をキーに、永続層から複数のデータを取得する。
	 *
	 * @param keys 検索キー
	 * @param allows[0] trueの場合、値が存在しないカラムも検索に一致したと見なす。可変パラメータのため設定しなくても良い。
	 * @param allows[1] trueの場合、先頭に#が存在する行をコメントと見なし処理対象としない。可変パラメータのため設定しなくても良い。
	 * @return Entityコレクション
	 */
	public Collection<E> findMany(final Collection<String> keys, final boolean... allows) {
		if (mCsv == null) {
			loadResource();
		}
		if (mCsv == null) throw new IllegalArgumentException("master csv isn't load.");

		Csv dstCsv = CsvUtil.select(mCsv, keys, mCompMap, allows);

		List<E> entityList = new ListNotNullDecorator<E>(new ArrayList<E>());
		for (List<String> rowList: dstCsv.mRowList) {
			entityList.add(this.createEntity(rowList));
		}
		return entityList;
	}

	/**
	 * 永続層からリソースを取得する。
	 */
	protected abstract void loadResource();

	/**
	 * Entityを登録する。
	 *
	 * @param entity Entity
	 */
	public <E2 extends E> void regist(final E2 entity) {
		if (mAbstFile == null && mFile == null) throw new IllegalStateException("The target isn't set.");

		try {
			PrintStream ps = new PrintStream(mAbstFile.openOutputStream(""), false, configDefaultCharacterEncoding());
			String row = this.createRow(StructUtil.createStrings(entity));
			ps.println(row);
			ps.close();
		} catch (UnsupportedEncodingException e) {
			throw new RuntimeException(e);
		}

		if (mFile != null) write(mAbstFile, mFile);
		if (mOutputStream != null) write(mAbstFile, mOutputStream);
	}

	/**
	 * Entityコレクションを登録する。
	 *
	 * @param entities Entityコレクション
	 */
	public <E2 extends E> void regist(final Collection<E2> entities) {
		if (mAbstFile == null && mFile == null) throw new IllegalStateException("The target isn't set.");

		try {
			PrintStream ps = new PrintStream(mAbstFile.openOutputStream(""), false, configDefaultCharacterEncoding());
			for (E2 entity: entities) {
				String row = this.createRow(StructUtil.createStrings(entity));
				ps.println(row);
			}
			ps.close();
		} catch (UnsupportedEncodingException e) {
			throw new RuntimeException(e);
		}

		if (mFile != null) write(mAbstFile, mFile);
		if (mOutputStream != null) write(mAbstFile, mOutputStream);
	}

	/**
	 * カラム配列から行文字列を生成する。
	 *
	 * @param cols カラム配列
	 * @return 行文字列
	 */
	private String createRow(final String[] cols) {
		Collection<Field> fields = mEntityClazzInfo.getFields();
		List<GdQuote> annoQuoteList = new ArrayList<GdQuote>();
		List<GdByteLength> annoByteLengthList = new ArrayList<GdByteLength>();
		for (Field field: fields) {
			annoQuoteList.add(field.getAnnotation(GdQuote.class));
			annoByteLengthList.add(field.getAnnotation(GdByteLength.class));
		}

		StringBuilder buf = new StringBuilder();
		int count = 0;
		for (String col: cols) {
			GdByteLength annoByteLength = annoByteLengthList.get(count);
			if (annoByteLength != null && annoByteLength.value() >= 0) {
				col = StringUtil.getLimitString(col, annoByteLength.value());
			}

			Character quote = null;
			GdQuote annoQuote = annoQuoteList.get(count);
			if (annoQuote != null) {
				if (annoQuote.add()) quote = annoQuote.type();
			} else {
				if (!NumberUtils.isDigits(col)) quote = '"';
			}

			col = escape(col);
			if (col != null) {
				if (quote != null) buf.append(quote);
				buf.append(col);
				if (quote != null) buf.append(quote);
			}
			if (count == cols.length - 1) break;
			count++;
			buf.append(',');
		}
		return buf.toString();
	}

	/**
	 * エスケープ処理を行う。
	 *
	 * @param src 対象文字列
	 * @return エスケープされた文字列
	 */
	protected String escape(String src) {
		return src.replaceAll("\"", "\"\"");
	}

	/**
	 * ファイルへ出力する。
	 *
	 * @param abstFile AbstFile
	 * @param target 書込ファイル
	 */
	private void write(AbstFile abstFile, File target) {
		FileOutputStream os = null;
		try {
			if (target.exists() && !target.canWrite()) {
				throw new RuntimeException("can't write " + target.getName() + ".");
			}

			os = new FileOutputStream(target);
			write(abstFile, os);
		} catch (Exception e) {
			throw new RuntimeException(e);
		} finally {
			FileIOUtil.close(os);
		}
	}

	/**
	 * OutputStreamへ出力する。
	 *
	 * @param abstFile AbstFile
	 * @param os OutputStream
	 */
	private void write(AbstFile abstFile, OutputStream os) {
		try {
			// 下記呼び出しでEOFException発生！！
			abstFile.write(os);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 列文字列配列からEntityを生成する。<br/>
	 * このメソッドは、Entityから情報を取得し、自動的に列の値を設定するが、<br/>
	 * 独自の処理を行いたい場合は、このメソッドをオーバーライドする。<br/>
	 *
	 * @return Entity
	 */
	protected E createEntity(final String[] cols) {
		return createEntity(Arrays.asList(cols));
	}

	/**
	 * 列文字列コレクションからEntityを生成する。<br/>
	 * このメソッドは、Entityから情報を取得し、自動的に列の値を設定するが、<br/>
	 * 独自の処理を行いたい場合は、このメソッドをオーバーライドする。<br/>
	 *
	 * @return Entity
	 */
	@SuppressWarnings("unchecked")
	protected E createEntity(final Collection<String> cols) {
		return StructUtil.createStruct((Class<E>)mEntityClazzInfo.getType(), cols, GdProperty.class);
	}



} // end-class
