/*
 * Copyright (c) 2007-2013 KITec Inc,.. All rights reserved.
 */
package option.gad.core.dxo;

import java.awt.Color;
import java.lang.reflect.Array;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import jp.kitec.lib.util.ClassInfo;
import jp.kitec.lib.util.ClassInfoManager;
import jp.kitec.lib.util.ListNotNullDecorator;
import jp.kitec.lib.util.ReflectUtil;
import jp.kitec.lib.util.StringUtil;

import org.apache.commons.lang.math.NumberUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;



/**
 * 型変換ユーティリティクラス
 *
 * @author $Author$
 * @version $Revision$ $Date::                           $
 */
public class TypeConvertUtil {

	//-----------------------------------------------------------------------
	//- fields
	//-----------------------------------------------------------------------

	/** ログ */
	private static final Log mLog = LogFactory.getLog(TypeConvertUtil.class);



	//------------------------------------------------------------------
	//- constructors
	//------------------------------------------------------------------

	/**
	 * コンストラクタ
	 */
	protected TypeConvertUtil() {
		super();
	}



	//------------------------------------------------------------------
	//- methods
	//------------------------------------------------------------------

	/**
	 * オブジェクトを任意のクラスに変換する。<br/>
	 *
	 * @param dstClazz 変換したいクラス
	 * @param src 変換対象
	 * @return 変換されたオブジェクト
	 */
	@SuppressWarnings("unchecked")
	public static <T,U> U convert(final Class<U> dstClazz, final T src) {
		if (src == null) return null;
		U result = null;

		if (src.getClass().isArray()) {
			ClassInfo clazzInfo = ClassInfoManager.getInstance().getClassInfo(dstClazz);
			Class<?> dstClazz2 = clazzInfo.getArrayType();

			Object[] workArray = (Object[])src;
			List<Object> dstList = new ListNotNullDecorator<Object>(new ArrayList<Object>());
			for (Object work: workArray) {
				dstList.add(convertObject(dstClazz2, convertString(work)));
			}
			result = (U)dstList.toArray((Object[])Array.newInstance(dstClazz2, dstList.size()));
		} else {
			result = convertObject(dstClazz, convertString(src));
		}

		return result;
	}

	/**
	 * オブジェクトを文字列に変換する。<br/>
	 *
	 * @param src 変換対象
	 * @return 変換された文字列
	 */
	public static String convertString(final Object src) {
		if (src == null) return null;

		String result = null;

		try {
			if (src.getClass() == boolean.class) {
				result = convertString((Boolean)src);
			} else if (src.getClass() == byte.class) {
				result = convertString(src);
			} else if (src.getClass() == int.class) {
				result = convertString((Integer)src);
			} else if (src.getClass() == long.class) {
				result = convertString(src);
			} else if (src.getClass() == float.class) {
				result = convertString((Float)src);
			} else if (src.getClass() == double.class) {
				result = convertString(src);
			} else if (Boolean.class.isAssignableFrom(src.getClass())) {
				result = convertString((Boolean)src);
			} else if (Byte.class.isAssignableFrom(src.getClass())) {
				result = convertString(src);
			} else if (Integer.class.isAssignableFrom(src.getClass())) {
				result = convertString((Integer)src);
			} else if (Long.class.isAssignableFrom(src.getClass())) {
				result = convertString(src);
			} else if (Float.class.isAssignableFrom(src.getClass())) {
				result = convertString((Float)src);
			} else if (Double.class.isAssignableFrom(src.getClass())) {
				result = convertString(src);
			} else if (Color.class.isAssignableFrom(src.getClass())) {
				result = convertString((Color)src);
			} else if (Timestamp.class.isAssignableFrom(src.getClass())) {
				result = convertString((Timestamp)src);
			} else if (String.class.isAssignableFrom(src.getClass())) {
				result = convertString((String)src);
			} else if (src.getClass().isEnum()) {
				result = convertString((Enum<?>)src);
			} else if (src instanceof GdTypeConvertable<?>) {
				result = convertString((GdTypeConvertable<?>)src);
			} else {
				mLog.debug("src[" + src + "] class[" + src.getClass() + "]");
			}
		} catch (Exception e) {
			mLog.info("", e);
		}

		return result;
	}

	/**
	 * 値を文字列に変換する。
	 *
	 * @param src 変換対象
	 * @return 変換された文字列
	 */
	public static String convertString(final Boolean src) {
		return (src == null)? null: String.valueOf(src);
	}

	/**
	 * 値を文字列に変換する。<br/>
	 * 小数が0の場合、小数は取り除きます。ex:1.0 --> 1<br/>
	 *
	 * @param src 変換対象
	 * @return 変換された文字列
	 */
	public static String convertString(final Float src) {
		if (src == null) return null;

		String str = String.valueOf(src);
		char decimal = str.charAt(str.length() - 1);
		return decimal == '0'? String.valueOf(src.intValue()): str;
	}

	/**
	 * 値を文字列に変換する。
	 *
	 * @param src 変換対象
	 * @return 変換された文字列
	 */
	public static String convertString(final Integer src) {
		return (src == null)? null: String.valueOf(src);
	}

	/**
	 * 値を文字列に変換する。
	 *
	 * @param src 変換対象
	 * @return 変換された文字列
	 */
	public static String convertString(final Color src) {
		return (src == null)? null: String.valueOf(src.getRGB());
	}

	/**
	 * 値を文字列に変換する。
	 *
	 * @param src 変換対象
	 * @return 変換された文字列
	 */
	public static String convertString(final Timestamp src) {
		return (src == null)? null: String.valueOf(src.getTime());
	}

	/**
	 * Enumを文字列に変換する。
	 *
	 * @param src 変換対象
	 * @return 変換された文字列
	 */
	public static String convertString(final Enum<?> src) {
		return (src == null)? null: src.name();
	}

	/**
	 * Enumコレクションを文字列に変換する。
	 *
	 * @param src 変換対象
	 * @return 変換された文字列
	 */
	public static <T extends Enum<T>> String convertString(final Collection<T> srcs) {
		if (srcs == null) return null;

		List<String> strList = new ArrayList<String>();
		for (T src: srcs) {
			strList.add(convertString(src));
		}
		return StringUtil.concat(strList, ",");
	}

	/**
	 * Stringを文字列に変換する。<br/>
	 * (スルーするだけ)<br/>
	 *
	 * @param src 変換対象
	 * @return 変換された文字列
	 */
	public static String convertString(final String src) {
		return (src == null)? null: src;
	}

	/**
	 * Convertableを文字列に変換する。
	 *
	 * @param src 変換対象
	 * @return 変換された文字列
	 */
	public static String convertString(final GdTypeConvertable<?> src) {
		if (src == null) return null;
		GdTypeConvertable<?> typeConv = src;
		return typeConv.toString();
	}

	/**
	 * 文字列を適切なオブジェクトに変換する。
	 *
	 * @param src 変換対象
	 * @return 適切なオブジェクト
	 */
	@SuppressWarnings("unchecked")
	public static <T> T convertObject(final Class<T> clazz, final String src) {
		if (StringUtil.isEmpty(src)) return null;

		T result = null;
		try {
			if (clazz == boolean.class) {
				result = (T)convertBoolean(src);
			} else if (clazz == byte.class) {
				result = (T)convertByte(src);
			} else if (clazz == int.class) {
				result = (T)convertInteger(src);
			} else if (clazz == long.class) {
				result = (T)convertLong(src);
			} else if (clazz == float.class) {
				result = (T)convertFloat(src);
			} else if (clazz == double.class) {
				result = (T)convertDouble(src);
			} else if (Boolean.class.isAssignableFrom(clazz)) {
				result = (T)convertBoolean(src);
			} else if (Byte.class.isAssignableFrom(clazz)) {
				result = (T)convertByte(src);
			} else if (Integer.class.isAssignableFrom(clazz)) {
				result = (T)convertInteger(src);
			} else if (Long.class.isAssignableFrom(clazz)) {
				result = (T)convertLong(src);
			} else if (Float.class.isAssignableFrom(clazz)) {
				result = (T)convertFloat(src);
			} else if (Double.class.isAssignableFrom(clazz)) {
				result = (T)convertDouble(src);
			} else if (Color.class.isAssignableFrom(clazz)) {
				result = (T)convertColor(src);
			} else if (Timestamp.class.isAssignableFrom(clazz)) {
				result = (T)convertTimestamp(src);
			} else if (String.class.isAssignableFrom(clazz)) {
				result = (T)convertString(src);
			} else if (clazz.isEnum()) {
				result = (T)convertEnum(clazz, src);
			} else if (GdTypeConvertable.class.isAssignableFrom(clazz)) {
				result = (T)convertTypeConvertable(clazz, src);
			} else {
				mLog.debug("src[" + src + "] srcClass[" + src.getClass() + "] toClass[" + clazz + "]");
			}
		} catch (Exception e) {
			mLog.info("", e);
		}

		return result;
	}

	/**
	 * 文字列をBooleanに変換する。
	 *
	 * @param src 変換対象
	 * @return Boolean
	 */
	public static Boolean convertBoolean(final String src) {
		Boolean result = null;
		if (NumberUtils.isNumber(src)) {
			int num = NumberUtils.toInt(src);
			if (num == 0) result = false;
			if (num == 1) result = true;
		} else {
			if (src.equalsIgnoreCase("false")) result = false;
			if (src.equalsIgnoreCase("true")) result = true;
		}
		return result;
	}

	/**
	 * 文字列をByteに変換する。
	 *
	 * @param src 変換対象
	 * @return Byte
	 */
	public static Byte convertByte(final String src) {
		Byte result = null;
		if (NumberUtils.isNumber(src)) {
			result = Byte.parseByte(src);
		}
		return result;
	}

	/**
	 * 文字列をIntegerに変換する。
	 *
	 * @param src 変換対象
	 * @return Integer
	 */
	public static Integer convertInteger(final String src) {
		Integer result = null;
		if (NumberUtils.isNumber(src)) {
			result = Integer.parseInt(src);
		}
		return result;
	}

	/**
	 * 文字列をLongに変換する。
	 *
	 * @param src 変換対象
	 * @return Long
	 */
	public static Long convertLong(final String src) {
		Long result = null;
		if (NumberUtils.isNumber(src)) {
			result = Long.parseLong(src);
		}
		return result;
	}

	/**
	 * 文字列をFloatに変換する。
	 *
	 * @param src 変換対象
	 * @return Float
	 */
	public static Float convertFloat(final String src) {
		Float result = null;
		if (NumberUtils.isNumber(src)) {
			result = Float.parseFloat(src);
		}
		return result;
	}

	/**
	 * 文字列をDoubleに変換する。
	 *
	 * @param src 変換対象
	 * @return Double
	 */
	public static Double convertDouble(final String src) {
		Double result = null;
		if (NumberUtils.isNumber(src)) {
			result = Double.parseDouble(src);
		}
		return result;
	}

	/**
	 * 文字列をColorに変換する。
	 *
	 * @param src 変換対象
	 * @return Color
	 */
	public static Color convertColor(final String src) {
		Color result = null;
		try {
			result = new Color(Integer.parseInt(src, 16));
		} catch (NumberFormatException e) {
		}
		return result;
	}

	/**
	 * 文字列をTimestampに変換する。
	 *
	 * @param src 変換対象
	 * @return Timestamp
	 */
	public static Timestamp convertTimestamp(final String src) {
		return new Timestamp(Long.parseLong(src));
	}

	/**
	 * 文字列をEnumに変換する。
	 *
	 * @param clazz 変換するEnumクラス
	 * @param src 変換対象
	 * @return Enum
	 */
	@SuppressWarnings("unchecked")
	public static Object convertEnum(final Class<?> clazz, final String src) {
		if (StringUtil.isEmpty(src)) return null;

		Object result = null;
		try {
			@SuppressWarnings("rawtypes")
			Class<Enum> enumClazz = (Class<Enum>)clazz;
			result = Enum.valueOf(enumClazz, src);
		} catch (Exception e) {
			mLog.warn("", e);
		}
		return result;
	}

	/**
	 * 文字列をGdTypeConvertableに変換する。
	 *
	 * @param src 変換対象
	 * @return GdTypeConvertable
	 */
	public static Object convertTypeConvertable(final Class<?> clazz, final String src) {
		if (StringUtil.isEmpty(src)) return null;

		Object result = null;
		try {
			@SuppressWarnings("unchecked")
			Class<GdTypeConvertable<?>> convClazz = (Class<GdTypeConvertable<?>>)clazz;
			GdTypeConvertable<?> typeConv = ReflectUtil.newInstance(convClazz);
			result = typeConv.toType(src);
		} catch (Exception e) {
			mLog.warn("", e);
		}
		return result;
	}



} // end-class
