/*
 * Copyright (c) 2009-2014 KITec Inc,.. All rights reserved.
 */
package option.gad.core.util.csv;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;

import jp.kitec.lib.util.StringUtil;



/**
 * Csvユーティリティクラス
 *
 * @author $Author$
 * @version $Revision$ $Date::                           $
 */
public class CsvUtil {

	/**
	 * コンストラクタ
	 */
	protected CsvUtil() {
	}



	/** 比較ロジックMap */
	public static Map<Class<?>, Comparator<String>> comparatorMap = new HashMap<Class<?>, Comparator<String>>();

	static {
		comparatorMap.put(StringComparator.class, new StringComparator());
		comparatorMap.put(BitFieldComparator.class, new BitFieldComparator());
		comparatorMap.put(GreaterThanComparator.class, new GreaterThanComparator());
		comparatorMap.put(GreaterEqualComparator.class, new GreaterEqualComparator());
		comparatorMap.put(LessThanComparator.class, new LessThanComparator());
		comparatorMap.put(LessEqualComparator.class, new LessEqualComparator());
	}



	/**
	 * 検索キーと一致したCSVデータを作成する。
	 *
	 * @param srcCsv 比較対象CSVデータ
	 * @param keyList 検索キーリスト
	 * @param allows[0] trueの場合、値が存在しないカラムも検索に一致したと見なす。可変パラメータのため設定しなくても良い。
	 * @param allows[1] trueの場合、先頭に#が存在する行をコメントと見なし処理対象としない。可変パラメータのため設定しなくても良い。
	 * @return 検索一致CSVデータ
	 * @throws IllegalStateException
	 */
	public static Csv select(Csv srcCsv, Collection<String> keyList, boolean... allows) {
		return select(srcCsv, keyList, null, allows);
	}

	/**
	 * 検索キーと一致したCSVデータを作成する。
	 *
	 * @param srcCsv 比較対象CSVデータ
	 * @param keyList 検索キーリスト
	 * @param compMap カラム毎の比較処理。結果が0の場合、同等とみなす。
	 * @param allows[0] trueの場合、値が存在しないカラムも検索に一致したと見なす。可変パラメータのため設定しなくても良い。
	 * @param allows[1] trueの場合、先頭に#が存在する行をコメントと見なし処理対象としない。可変パラメータのため設定しなくても良い。
	 * @return 検索一致CSVデータ
	 * @throws IllegalStateException
	 */
	public static Csv select(
			Csv srcCsv, Collection<String> keyList, Map<String, Comparator<String>> compMap, boolean... allows) {
		if (keyList.size() != srcCsv.mColNameList.size())
			throw new IllegalArgumentException(
				"keyList.size[" + keyList.size() + "] colNameList.size[" + srcCsv.mColNameList.size() + "]");

		Csv dstCsv = new Csv();
		for (List<String> colList: srcCsv.mRowList) {
			if (!match(srcCsv.mColNameList, keyList, colList, compMap, allows)) continue;

			dstCsv.mRowList.add(new ArrayList<String>(colList));
		}
		return dstCsv;
	}

	/**
	 * 検索キーリストとカラム値リストが一致したかを判断する。
	 *
	 * @param keyList 検索キーリスト
	 * @param colList カラム値リスト
	 * @param compMap カラム毎の比較処理。結果が0の場合、同等とみなす。
	 * @param allows[0] trueの場合、値が存在しないカラムも検索に一致したと見なす。可変パラメータのため設定しなくても良い。
	 * @param allows[1] trueの場合、先頭に#が存在する行をコメントと見なし処理対象としない。可変パラメータのため設定しなくても良い。
	 * @return 条件が一致した場合はtrue。
	 * @throws IllegalStateException
	 */
	private static boolean match(
			Collection<String> nameList, Collection<String> keyList, Collection<String> colList,
			Map<String, Comparator<String>> compMap, boolean... allows) {
		if (allows.length > 2) throw new IllegalArgumentException();
		boolean allowNull = (allows.length >= 1)? allows[0]: false;
		boolean allowComment = (allows.length >= 2)? allows[1]: false;

		for (Iterator<String>
				nameIte = nameList.iterator(),
				keyIte = keyList.iterator(),
				colIte = colList.iterator();
				keyIte.hasNext(); ) {
			String name = nameIte.next();
			String key = keyIte.next();
			String col = colIte.next();

			if (allowComment && !StringUtil.isEmpty(col) && col.charAt(0) == '#') return false;
			if (key == null) continue;

			if (allowNull && (col == null || col.length() == 0)) continue;

			Comparator<String> comp = null;
			if (compMap != null) comp = compMap.get(name);
			if (comp == null) comp = comparatorMap.get(StringComparator.class);

			if (comp.compare(key, col) != 0) return false;
		}
		return true;
	}

	/**
	 * 文字列比較で同一かを確認する
	 *
	 * @param key 検索キー
	 * @param csv CSVデータ
	 */
	public static class StringComparator implements Comparator<String> {
		public int compare(String key, String csv) {
			return key.equals(csv)? 0: -1;
		}
	};

	/**
	 * ビットフィールドの比較処理<br/>
	 * 比較対象をビットフィールドとみなし、&演算子で比較を行う。<br/>
	 *
	 * @param key 検索キー
	 * @param csv CSVデータ
	 */
	public static class BitFieldComparator implements Comparator<String> {
		public int compare(String key, String csv) {
			Integer ikey = Integer.parseInt(key);
			Integer icsv = Integer.parseInt(csv);
			return ((ikey & icsv) != 0)? 0: -1;
		}
	};

	/**
	 * keyがcsvより大きいかを確認する
	 *
	 * @param key 検索キー
	 * @param csv CSVデータ
	 */
	public static class GreaterThanComparator implements Comparator<String> {
		public int compare(String key, String csv) {
			Double dkey = Double.parseDouble(key);
			Double dcsv = Double.parseDouble(csv);
			return (dkey > dcsv)? 0: -1;
		}
	};

	/**
	 * keyがcsvより大きいか同じかを確認する
	 *
	 * @param key 検索キー
	 * @param csv CSVデータ
	 */
	public static class GreaterEqualComparator implements Comparator<String> {
		public int compare(String key, String csv) {
			Double dkey = Double.parseDouble(key);
			Double dcsv = Double.parseDouble(csv);
			return (dkey >= dcsv)? 0: -1;
		}
	};

	/**
	 * keyがcsvより小さいかを確認する
	 *
	 * @param key 検索キー
	 * @param csv CSVデータ
	 */
	public static class LessThanComparator implements Comparator<String> {
		public int compare(String key, String csv) {
			Double dkey = Double.parseDouble(key);
			Double dcsv = Double.parseDouble(csv);
			return (dkey < dcsv)? 0: -1;
		}
	};

	/**
	 * keyがcsvより小さいか同じかを確認する
	 *
	 * @param key 検索キー
	 * @param csv CSVデータ
	 */
	public static class LessEqualComparator implements Comparator<String> {
		public int compare(String key, String csv) {
			Double dkey = Double.parseDouble(key);
			Double dcsv = Double.parseDouble(csv);
			return (dkey <= dcsv)? 0: -1;
		}
	};

	/**
	 * 指定された文字列を、カンマで分割する。<BR>
	 * また、ダブルクォートで囲まれた文字列内に存在する
	 * カンマは無視する。
	 *
	 * @param		str		分割する文字列
	 * @return	指定された文字列をカンマに
	 * 				一致する位置で分割して計算された文字列の配列
	 * @throws	NoSuchElementException	不正にダブルクォートが
	 * 										使用されていた場合
	 */
	public static String[] split(String str) throws NoSuchElementException {
		return split(str, ",", "\"");
	}

	/**
	 * 指定された文字列を、指定された区切り文字で分割する。<BR>
	 * また、引用符で囲まれた文字列内に存在する
	 * 区切り文字は無視する。
	 *
	 * @param		str		分割する文字列
	 * @param 		delim	区切り文字
	 * @param		quote	引用符
	 * @return	指定された文字列を指定された区切り文字に
	 * 				一致する位置で分割して計算された文字列の配列
	 * @throws	NoSuchElementException	不正に引用符が
	 * 										使用されていた場合
	 */
	public static String[] split(String str,
								final String delim,
								final String quote)
								throws NoSuchElementException {
		final int st = str.indexOf(quote);
		final int ed = str.lastIndexOf(quote);
		if (st < 0 || st >= ed) {
			return str.split(delim);
		}

		List<String> tokenList = new ArrayList<String>();

		int current = 0, d, q;
		int onQuoted = 0;
		final int len = str.length();

		while (current < len) {
			d = str.indexOf(delim, current);
			q = str.indexOf(quote, current);

			if (onQuoted == 1) {
				if (q < 0) {
				throw new NoSuchElementException(
						"illegal quotation [" + str + "]");
				}
				else {
				tokenList.add(str.substring(current, q));
				current     = q + 1;
				onQuoted    = 2;
				continue;
				}
			}

			if (d < 0 && q < 0) {
				String s = str.substring(current, len).trim();
				if (s.length() > 0) {
					if (onQuoted == 2) {
						throw new NoSuchElementException(
								"illegal quotation [" + str + "]");
					}
					if (current < len) {
						tokenList.add(s);
					}
				}
				break;
			}

			if (d >= 0) {
				if (q < 0 || d < q) {
					if (onQuoted == 2) {
						onQuoted = 0;
					}
					else {
						tokenList.add(str.substring(current, d).trim());
					}
					current = d + 1;
					continue;
				}
			}

			if (q >= 0) {
				current = q + 1;
				onQuoted = 1;
			}
		}

		String[] tokens = new String[tokenList.size()];
		for (int i = 0; i < tokens.length; i++) {
			tokens[i] = tokenList.get(i);
		}
		return tokens;
	}



} // end-class
