/*
 * Copyright (c) 2009-2010 KITec Inc,.. All rights reserved.
 */
package option.gad.core.dao;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.lang.reflect.Field;

import jp.kitec.lib.io.AbstFile;
import jp.kitec.lib.io.AbstFileUtil;
import jp.kitec.lib.util.ClassInfo;
import jp.kitec.lib.util.ClassInfoManager;
import jp.kitec.lib.util.NameUtil;
import jp.kitec.lib.util.ReflectUtil;
import jp.kitec.lib.util.StringUtil;
import option.gad.core.dxo.TypeConvertUtil;
import option.gad.core.entity.GdEntity;
import option.gad.core.io.FileIOUtil;



/**
 * KITec独自フォーマットDaoの抽象実装
 *
 * @author $Author$
 * @version $Revision$ $Date::                           $
 */
public abstract class AbstractKitecFormatDao<E> {

	//------------------------------------------------------------------
	//- fields
	//------------------------------------------------------------------

	/** リソース名 */
	protected String mResourceName = null;

	/** Entityクラス情報 */
	protected ClassInfo mEntityClazzInfo = null;

	/** セクション名 */
	protected String mSectionName = null;

	protected SectionManager mSectionManager = null;

	protected AbstFile mAbstFile = null;

	protected File mFile = null;

	protected OutputStream mOutputStream = null;

	protected BufferedReader mReader = null;


	//------------------------------------------------------------------
	//- constructors
	//------------------------------------------------------------------

	/**
	 * コンストラクタ
	 */
	protected AbstractKitecFormatDao() {
		super();
		this.init();
	}



	//------------------------------------------------------------------
	//- initialize methods
	//------------------------------------------------------------------

	/**
	 * 初期化する。
	 */
	private void init() {
		ClassInfo classInfo = ClassInfoManager.getInstance().getClassInfo(this.getClass());

		GdDao annoDao = classInfo.getType().getAnnotation(GdDao.class);
		if (annoDao == null) throw new IllegalStateException("GdDao is not defined in " + classInfo.getType().getName() + ".");
		this.mEntityClazzInfo = ClassInfoManager.getInstance().getClassInfo(annoDao.entity());

		GdEntity annoEntity = this.mEntityClazzInfo.getType().getAnnotation(GdEntity.class);
		if (annoEntity == null) throw new IllegalStateException("GdEntity is not defined in " + this.mEntityClazzInfo.getType().getName() + ".");
		this.mSectionName = annoEntity.sectionName();
	}



	//------------------------------------------------------------------
	//- methods
	//------------------------------------------------------------------

	/**
	 * 永続層からデータを取得する。<br/>
	 * データが取得できなかった場合は、nullを返す。<br/>
	 *
	 * @return Entity
	 */
	public E findOne() {
		if (mFile == null) return null;
		if (!mFile.exists()) return null;
		if (!mFile.canRead()) return null;

		if (!AbstFileUtil.read(mAbstFile, mFile)) throw new RuntimeException();
		mReader = mAbstFile.openBufferedReader();
		if (mReader == null) throw new RuntimeException();

		if (mSectionManager == null) loadResource();
		Section section = mSectionManager.searchSection(mSectionName);
		if (section == null) section = mSectionManager.searchSection(SectionManager.DEF_ANONYMOUS);
		if (section == null) throw new RuntimeException("SectionName not found.");

		E entity = createEntity();
		settingDataToEntity(entity, section);
		return entity;
	}

	/**
	 * Entityを生成する。
	 *
	 * @return Entity
	 */
	@SuppressWarnings("unchecked")
	private E createEntity() {
		return (E)ReflectUtil.newInstance(this.mEntityClazzInfo.getType());
	}

	/**
	 * セクションデータをEntityに設定する。
	 *
	 * @param entity Entity
	 * @param section セクション
	 */
	private void settingDataToEntity(final E entity, final Section section) {
		final int count = section.getDataCount();

		for (int i = 0; i < count; i++) {
			String row = section.getData(i);
			String[] cols = StringUtil.split(row, ",", "\"");
			String fieldName = NameUtil.addPrefix(cols[0], 'm');
			String value = null;
			if (cols.length > 1) value = cols[1];
			ReflectUtil.setFieldValue(entity, fieldName, value);
		}
	}

	protected abstract void loadResource();

	/**
	 * Entityを永続層に書き込む。
	 *
	 * @param entity Entity
	 */
	public void regist(final E entity) {
		PrintStream ps = new PrintStream(mAbstFile.openOutputStream(""));
		if (!StringUtil.isEmpty(mSectionName)) {
			ps.println("[" + mSectionName + "]");
		}
		this.settingEntityToAbstFile(entity, ps);

		ps.close();

		if (mFile != null) write(mAbstFile, mFile);
		if (mOutputStream != null) {
			try {
				mAbstFile.write(mOutputStream);
			} catch (IOException e) {
				throw new RuntimeException(e);
			}
		}
	}

	/**
	 * EntityデータをAbstFileに設定する。
	 *
	 * @param entity Entity
	 * @param abstFile AbstFile
	 */
	private void settingEntityToAbstFile(final Object entity, final PrintStream ps) {
		for (Field field: ReflectUtil.getFields(entity.getClass())) {
			String itemName = NameUtil.removePrefix(field.getName());
			Object fieldValue = ReflectUtil.getFieldValue(entity, field);
			String strFieldValue = TypeConvertUtil.convert(String.class, fieldValue);
			strFieldValue = StringUtil.emptyToZeroString(strFieldValue);
			ps.println(itemName + ",\"" + strFieldValue + "\"");
		}
	}

	/**
	 * ファイルへ出力する。
	 *
	 * @param abstFile AbstFile
	 * @param target 書込ファイル
	 */
	private void write(AbstFile abstFile, File target) {
		FileOutputStream os = null;
		try {
			if (target.exists() && !target.canWrite()) {
				throw new RuntimeException("can't write " + target.getName() + ".");
			}

			os = new FileOutputStream(target);
			//	下記呼び出しでEOFException発生！！
			abstFile.write(os);
		} catch (Exception e) {
			throw new RuntimeException(e);
		} finally {
			FileIOUtil.close(os);
		}
	}



} // end-class
