/*
 * Copyright (c) 2007-2014 KITec Inc,.. All rights reserved.
 */
package option.gad.core.io;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.util.zip.GZIPInputStream;

import jp.kitec.lib.util.Closure;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;



/**
 * 入出力系のファイルユーティリティクラスです。
 *
 * @author $Author$
 * @version $Revision$ $Date::                           $
 */
public class FileIOUtil {

	//------------------------------------------------------------------
	//- constants
	//------------------------------------------------------------------

	/** バッファサイズ */
	private static final int BUF_SIZE = 81920;



	//------------------------------------------------------------------
	//- fields
	//------------------------------------------------------------------

	/** ログ */
	private static final Log _log = LogFactory.getLog(FileIOUtil.class);



	//------------------------------------------------------------------
	//- constructors
	//------------------------------------------------------------------

	/**
	 * コンストラクタ（外部new禁止）
	 */
	protected FileIOUtil() {
	}



	//------------------------------------------------------------------
	//- methods
	//------------------------------------------------------------------

	/**
	 * テキストファイルを取得します。
	 *
	 * @param file 取得ファイル情報
	 * @param encodeType テキストファイルのエンコード
	 * @return テキスト内容
	 */
	public static String readText(File file, String encodeType) {
		String result = null;
		try {
			result = readText(new FileInputStream(file), encodeType);
		} catch (FileNotFoundException e) {
			_log.error("", e);
		}

		return result;
	}

	/**
	 * テキストを取得します。
	 *
	 * @param is InputStream
	 * @param encodeType テキストのエンコード
	 * @return テキスト内容
	 */
	public static String readText(InputStream is, String encodeType) {
		String result = null;
		Reader reader = null;
		try {
			reader = new BufferedReader(new InputStreamReader(is, encodeType));

			StringBuffer buf = new StringBuffer();
			while (true) {
				char[] cbuf = new char[BUF_SIZE];
				int size = reader.read(cbuf, 0, BUF_SIZE);
				if (size == -1) break;

				buf.append(cbuf, 0, size);
			}
			result = buf.toString();

		} catch (Exception e) {
			_log.error("", e);
		} finally {
			close(reader);
		}

		return result;
	}

	/**
	 * テキストを取得します。<br/>
	 * １行読み込むごとにClosureが実行されます。<br/>
	 *
	 * @param file 取得ファイル情報
	 * @param encodeType テキストのエンコード
	 * @param closure Closure
	 */
	public static void readText(File file, String encodeType, Closure<String, Boolean> closure) {
		try {
			InputStream inputStream = new FileInputStream(file);
			readText(inputStream, encodeType, closure);
		} catch (FileNotFoundException e) {
			_log.info("", e);
		}
	}

	/**
	 * テキストを取得します。<br/>
	 * １行読み込むごとにClosureが実行されます。<br/>
	 *
	 * @param file 取得ファイル情報
	 * @param encodeType テキストのエンコード
	 * @param closure Closure
	 */
	public static void readText(String path, String encodeType, Closure<String, Boolean> closure) {
		InputStream inputStream = FileIOUtil.class.getResourceAsStream(path);
		readText(inputStream, encodeType, closure);
	}

	/**
	 * テキストを取得します。<br/>
	 * １行読み込むごとにClosureが実行されます。<br/>
	 *
	 * @param inputStream InputStream
	 * @param encodeType テキストのエンコード
	 * @param closure Closure
	 */
	public static void readText(InputStream inputStream, String encodeType, Closure<String, Boolean> closure) {
		try {
			Reader reader = new InputStreamReader(inputStream, encodeType);
			readText(reader, closure);
		} catch (UnsupportedEncodingException e) {
			_log.info("", e);
		}
	}

	/**
	 * テキストを取得します。<br/>
	 * １行読み込むごとにClosureが実行されます。<br/>
	 *
	 * @param reader Reader
	 * @param closure Closure
	 */
	public static void readText(Reader reader, Closure<String, Boolean> closure) {
		BufferedReader br = null;
		try {
			br = new BufferedReader(reader);

			while (true) {
				String line = br.readLine();
				if (line == null) break;
				boolean next = closure.execute(line);
				if (!next) break;
			}

		} catch (Exception e) {
			_log.info("", e);
		} finally {
			FileIOUtil.close(br);
		}
	}

	/**
	 * テキストファイルを作成します。
	 *
	 * @param file 作成ファイル情報
	 * @param text テキスト
	 * @param encodeType テキストファイルのエンコード
	 * @return 処理が正常に終了した場合はtrue。
	 */
	public static boolean writeText(File file, String text, String encodeType) {
		boolean result = false;
		Writer writer = null;

		try {
			writer = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(file), encodeType));
			writer.write(text);
			writer.flush();

			result = true;
		} catch (Exception e) {
			_log.error("", e);
		} finally {
			close(writer);
		}

		return result;
	}

	/**
	 * ファイルを読込オープンします。
	 *
	 * @param file 読込ファイル情報
	 * @return 入力ストリーム
	 * @throws FileNotFoundException
	 */
	public static InputStream openInReadingMode(File file) throws FileNotFoundException {
		return new BufferedInputStream(new FileInputStream(file));
	}

	@Deprecated
	public static InputStream readOpen(File file) throws FileNotFoundException {
		return openInReadingMode(file);
	}

	/**
	 * Gzipファイルを読込オープンします。<br/>
	 * Gzipファイルでなかった場合、通常ファイルとしてオープンします。<br/>
	 *
	 * @param file 読込ファイル情報
	 * @return 入力ストリーム
	 */
	public static InputStream openGzipInReadingMode(File file) {
		InputStream is = null;

		try {
			is = new GZIPInputStream(new FileInputStream(file));
		} catch (Exception e) {
			try {
				is = new FileInputStream(file);
			} catch (Exception e2) {
				throw new RuntimeException(e);
			}
		}

		return is;
	}

	/**
	 * ファイルの取得を行います。
	 *
	 * @param file 取得ファイル情報
	 * @param outputStream 出力ストリーム
	 * @return 処理が正常に終了した場合はtrue。
	 */
	public static boolean read(File file, OutputStream outputStream) {
		boolean result = false;
		InputStream inputStream = null;

		try {
			inputStream = new BufferedInputStream(new FileInputStream(file));
			result = write(outputStream , inputStream);
		} catch (Exception e) {
			_log.error("", e);
		} finally {
			close(inputStream);
		}

		return result;
	}

	/**
	 * ファイルの作成を行います。
	 *
	 * @param file 作成ファイル情報
	 * @param inputStream 入力ストリーム
	 * @return 処理が正常に終了した場合はtrue。
	 */
	public static boolean write(File file, InputStream inputStream) {
		boolean result = false;
		OutputStream outputStream = null;

		try {
			outputStream = new BufferedOutputStream(new FileOutputStream(file));
			result = write(outputStream , inputStream);
		} catch (Exception e) {
			_log.error("", e);
		} finally {
			close(outputStream);
		}

		return result;
	}

	/**
	 * ストリームの入出力を行います。<br/>
	 * ストリームのクローズは行いません。<br/>
	 *
	 * @param outputStream 出力ストリーム
	 * @param inputStream 入力ストリーム
	 * @return 処理が正常に終了した場合はtrue。
	 */
	public static boolean write(OutputStream outputStream, InputStream inputStream) {
		boolean result = false;

		try {
			int bytesRead = 0;
			byte[] buffer = new byte[BUF_SIZE];
			while ((bytesRead = inputStream.read(buffer, 0, BUF_SIZE)) != -1) {
				outputStream.write(buffer, 0, bytesRead);
			}
			outputStream.flush();

			result = true;
		} catch (Exception e) {
			_log.error("", e);
		}

		return result;
	}

	/**
	 * ファイルをバイト列として取得します。
	 *
	 * @param file 取得ファイル情報
	 * @return ファイルのバイト列
	 */
	public static byte[] fileToBytes(File file) {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		read(file, baos);
		return baos.toByteArray();
	}

	/**
	 * バイト列をファイルに出力します。
	 *
	 * @param file 作成ファイル情報
	 * @param fileBytes ファイルに出力するバイト列
	 */
	public static void bytesToFile(File file, byte[] fileBytes) {
		ByteArrayInputStream bais = new ByteArrayInputStream(fileBytes);
		write(file, bais);
	}

	/**
	 * InputStreamをクローズします。
	 *
	 * @param inputStream InputStream
	 * @return 処理が正常に終了した場合はtrue。
	 */
	public static boolean close(InputStream inputStream) {
		boolean result = true;
		if (inputStream != null) {
			try {
				inputStream.close();
			} catch (IOException e) {
				result = false;
				_log.error("", e);
			}
			inputStream = null;
		}
		return result;
	}

	/**
	 * OutputStreamをクローズします。
	 *
	 * @param outputStream OutputStream
	 * @return 処理が正常に終了した場合はtrue。
	 */
	public static boolean close(OutputStream outputStream) {
		boolean result = true;
		if (outputStream != null) {
			try {
				outputStream.close();
			} catch (IOException e) {
				result = false;
				_log.error("", e);
			}
			outputStream = null;
		}
		return result;
	}

	/**
	 * Readerをクローズします。
	 *
	 * @param reader Reader
	 * @return 処理が正常に終了した場合はtrue。
	 */
	public static boolean close(Reader reader) {
		boolean result = true;
		if (reader != null) {
			try {
				reader.close();
			} catch (IOException e) {
				result = false;
				_log.error("", e);
			}
			reader = null;
		}
		return result;
	}

	/**
	 * Writerをクローズします。
	 *
	 * @param writer Writer
	 * @return 処理が正常に終了した場合はtrue。
	 */
	public static boolean close(Writer writer) {
		boolean result = true;
		if (writer != null) {
			try {
				writer.close();
			} catch (IOException e) {
				result = false;
				_log.error("", e);
			}
			writer = null;
		}
		return result;
	}



} // end-class
