/*
 * Copyright (c) 2008 KITec Inc,.. All rights reserved.
 */
package option.gad.core.inject;

import java.util.HashMap;
import java.util.Map;

import jp.kitec.lib.util.ReflectUtil;



/**
 * Framework用インスタンス生成クラス。
 *
 * @author $Author$
 * @version $Revision$ $Date::                           $
 */
public class GdInstanceFactory {

	//------------------------------------------------------------------
	//- fields
	//------------------------------------------------------------------

	/** インスタンス */
	protected static GdInstanceFactory mInstance = new GdInstanceFactory();

	/** インスタンス管理Map */
	protected Map<Class<?>, Object> mInstanceMap = new HashMap<Class<?>, Object>();

	/** スコープ管理Map */
	protected Map<Class<?>, GdScopeType> mScopeTypeMap = new HashMap<Class<?>, GdScopeType>();



	//------------------------------------------------------------------
	//- constructors
	//------------------------------------------------------------------

	/**
	 * コンストラクタ
	 */
	protected GdInstanceFactory() {
	}



	//------------------------------------------------------------------
	//- methods
	//------------------------------------------------------------------

	/**
	 * インスタンスを取得する。
	 *
	 * @return インスタンス
	 */
	public static GdInstanceFactory getInstance() {
		return mInstance;
	}

	/**
	 * インスタンスを取得する。
	 *
	 * @param clazz インスタンス生成対象クラス
	 * @param params クラスへ渡すパラメータ
	 * @return インスタンス
	 */
	public <T> T getInstance(final Class<T> clazz, Object... params) {
		T result = null;

		GdScopeType scopeType = this.mScopeTypeMap.get(clazz);
		if (scopeType == null) scopeType = GdScopeType.SINGLETON;

		switch (scopeType) {
		case SINGLETON: result = this.getSingletonInstance(clazz, params); break;
		case PROTOTYPE: result = this.getPrototypeInstance(clazz, params); break;
		}

		return result;
	}

	/**
	 * シングルトン・インスタンスを取得する。
	 *
	 * @param clazz インスタンス生成対象クラス
	 * @param params クラスへ渡すパラメータ
	 * @return インスタンス
	 */
	@SuppressWarnings("unchecked")
	public <T> T getSingletonInstance(final Class<T> clazz, final Object... params) {
		T instance = (T)this.mInstanceMap.get(clazz);
		if (instance == null) {
			instance = ReflectUtil.newInstance(clazz, params);
			this.mInstanceMap.put(clazz, instance);
			GdInjector.getInstance().clear();
			GdInjector.getInstance().inject(instance);
		}
		return instance;
	}

	/**
	 * プロトタイプ・インスタンスを取得する。
	 *
	 * @param clazz インスタンス生成対象クラス
	 * @param params クラスへ渡すパラメータ
	 * @return インスタンス
	 */
	public <T> T getPrototypeInstance(final Class<T> clazz, final Object... params) {
		T instance = ReflectUtil.newInstance(clazz, params);
		GdInjector.getInstance().clear();
		GdInjector.getInstance().inject(instance);
		return instance;
	}

	/**
	 * クラスのスコープを設定する。
	 *
	 * @param clazz
	 * @param scopeType
	 */
	public void addScope(final Class<?> clazz, final GdScopeType scopeType) {
		this.mScopeTypeMap.put(clazz, scopeType);
	}

	/**
	 * 保持内容をクリアする。
	 */
	public void clear() {
		mInstanceMap.clear();
		mScopeTypeMap.clear();
	}



} // end-class
